import React, { useState, useRef} from 'react';
import { useForm, usePage } from '@inertiajs/react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head } from '@inertiajs/react';
import { SwipeableButton } from "react-swipeable-button";
import HeroImg from "../../../images/hero-img.jpg";
import InputLabel from '@/Components/InputLabel';
import TextInput from '@/Components/TextInput';
import InputError from '@/Components/InputError';

export default function RedeemForm({ auth, games, settings }) {
    const pageProps = usePage().props;
    const { data, setData, post, processing, errors } = useForm({
        amount: '',
        game: '',
        username: '',
        network: 'lightning',
        address: '',
    });
    
    const [selectedGame, setSelectedGame] = useState(null);
    const [formErrors, setFormErrors] = useState({});
    const [showNetworkOptions, setShowNetworkOptions] = useState(false);
    const [isProcessing, setIsProcessing] = useState(false);
    
    const swipeableButtonRef = useRef();

    const handleReset = () => {
        swipeableButtonRef.current?.buttonReset();
    };

    const handleProceed = async () => {
        setFormErrors({});

        // Convert amount to number for proper comparison
        const amountNum = Number(data.amount);
        const minAmount = Number(settings.min_withdrawal_amount);
        const maxAmount = Number(settings.max_withdrawal_amount);

        if (!data.amount || amountNum < minAmount || amountNum > maxAmount) {
            setFormErrors((prev) => ({ 
                ...prev, 
                amount: `Amount must be between ${settings.min_withdrawal_amount} and ${settings.max_withdrawal_amount} ${settings.default_currency}.` 
            }));
            handleReset();
            return;
        }
        
        if (!selectedGame) {
            setFormErrors((prev) => ({ ...prev, game: "Please select a game." }));
            handleReset();
            return;
        }
        
        if (!data.username) {
            setFormErrors((prev) => ({ ...prev, username: "Please enter a username." }));
            handleReset();
            return;
        }

        // Show network options after validation passes
        setShowNetworkOptions(true);
    };

    const handleGameSelect = (gameId) => {
        setSelectedGame(gameId);
        setData('game', gameId);
    };

    const handleNetworkSelect = (network) => {
        setData('network', network);
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        if (isProcessing) {
            return; // Prevent multiple submissions
        }
        
        setIsProcessing(true);
        
        try {
            await post(route('redeem.process'), {
                onSuccess: () => {
                    setIsProcessing(false);
                },
                onError: () => {
                    setIsProcessing(false);
                }
            });
        } catch (error) {
            setIsProcessing(false);
        }
    };

    return (
        <AuthenticatedLayout user={auth.user} settings={settings}>
            <Head title="Redeem Points | VegasFish" />
            <div className="flex items-center justify-center min-h-[80vh]">
                <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-5 max-w-100 lg:max-w-[1020px] md:max-w-[600px]">
                    <div className="w-full p-4 py-[50px] bg-gray-100 dark:bg-gray-800 rounded-lg shadow-md">
                        <div className="text-center">
                            <h1 className="text-[28px] md:text-[35px] font-bold mb-4 inline-block font-bold bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">Redeem Your Points</h1>
                        </div>
                        {pageProps.errors?.error && <div className="text-red-500 text-[16px] text-center mb-2">{pageProps.errors?.error}</div>}
                        {!showNetworkOptions ? (
                            <form onSubmit={(e) => e.preventDefault()}>
                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">
                                        Enter Amount to Withdraw ({settings.default_currency})
                                    </label>
                                    <input
                                        type="number"
                                        value={data.amount}
                                        onChange={(e) => setData('amount', e.target.value)}
                                        className="w-full p-2 mt-1 border rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder="Enter Amount"
                                        min={settings.min_withdrawal_amount}
                                        max={settings.max_withdrawal_amount}
                                        step="1"
                                    />
                                    {formErrors.amount && <div className="text-red-500 text-sm">{formErrors.amount}</div>}
                                    {errors.amount && <div className="text-red-500 text-sm">{errors.amount}</div>}
                                </div>

                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">Enter Username</label>
                                    <input
                                        type="text"
                                        value={data.username}
                                        onChange={(e) => setData('username', e.target.value)}
                                        className="w-full p-2 mt-1 border rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder="Enter your game username"
                                    />
                                    {formErrors.username && <div className="text-red-500 text-sm">{formErrors.username}</div>}
                                    {errors.username && <div className="text-red-500 text-sm">{errors.username}</div>}
                                </div>

                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">Click below to select the Game</label>
                                    <div className="grid grid-cols-4 sm:grid-cols-6 md:grid-cols-6 lg:grid-cols-8 gap-3 md:gap-5 mt-2">
                                        {games.map((game) => (
                                            <div
                                                key={game.id}
                                                onClick={() => handleGameSelect(game.id)}
                                                className={`overflow-hidden border-2 shadow-md bg-white dark:bg-gray-700 cursor-pointer rounded-[15px] ${selectedGame === game.id ? 'border-[#FF6F00]' : 'border-transparent dark:border-transparent'} hover:border-[#FF6F00]`}
                                            >
                                                {game.game_image && (
                                                    <img
                                                        src={`storage/${game.game_image}`}
                                                        alt={game.game_name}
                                                        className="w-full object-cover"
                                                    />
                                                )}
                                            </div>
                                        ))}
                                    </div>
                                    {formErrors.game && <div className="text-red-500 text-sm">{formErrors.game}</div>}
                                    {errors.game && <div className="text-red-500 text-sm">{errors.game}</div>}
                                </div>

                                <div className="flex justify-center">
                                    <SwipeableButton
                                        ref={swipeableButtonRef}
                                        text="Swipe to Claim Prize"
                                        text_unlocked="Processing..."
                                        sliderColor="#960018"
                                        sliderTextColor="#fff"
                                        sliderIconColor="#fff"
                                        textColor="#000"
                                        background_color="#FF6F00"
                                        circle
                                        name="react-swipeable-button"
                                        onSuccess={handleProceed}
                                    />
                                </div>
                            </form>
                        ) : (
                            <div>
                                <h2 className="text-xl font-semibold mb-4 text-gray-800 dark:text-gray-200">
                                    Select Network and Enter Address
                                </h2>
                                
                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">Select Network</label>
                                    <div className="grid grid-cols-2 gap-4 mt-2">
                                        <div
                                            onClick={() => handleNetworkSelect('lightning')}
                                            className={`p-4 border rounded-lg cursor-pointer ${data.network === 'lightning' ? 'border-[#FF6F00] bg-[#FF6F00] dark:bg-[#960018]' : 'border-gray-300 dark:border-gray-600'}`}
                                        >
                                            <div className="text-center">
                                                <div className="text-lg font-semibold text-gray-800 dark:text-gray-200">Lightning Network</div>
                                                <div className="text-sm text-gray-600 dark:text-gray-400">Fast & Low Fees</div>
                                            </div>
                                        </div>
                                        <div
                                            onClick={() => handleNetworkSelect('onchain')}
                                            className={`p-4 border rounded-lg cursor-pointer ${data.network === 'onchain' ? 'border-[#FF6F00] bg-[#FF6F00] dark:bg-[#960018]' : 'border-gray-300 dark:border-gray-600'}`}
                                        >
                                            <div className="text-center">
                                                <div className="text-lg font-semibold text-gray-800 dark:text-gray-200">On-Chain</div>
                                                <div className="text-sm text-gray-600 dark:text-gray-400">Bitcoin Network</div>
                                            </div>
                                        </div>

                                        <div
                                            onClick={() => handleNetworkSelect('usdt')}
                                            className={`p-4 border rounded-lg cursor-pointer ${data.network === 'usdt' ? 'border-[#FF6F00] bg-[#FF6F00] dark:bg-[#960018]' : 'border-gray-300 dark:border-gray-600'}`}
                                        >
                                            <div className="text-center">
                                                <div className="text-lg font-semibold text-gray-800 dark:text-gray-200">USDT</div>
                                                <div className="text-sm text-gray-600 dark:text-gray-400">USDT Network</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">
                                        {data.network === 'lightning' ? 'Enter Lightning Invoice' : data.network === 'usdt' ? 'Enter USDT Invoice' : 'Enter Bitcoin Address'}
                                    </label>
                                    <textarea
                                        value={data.address}
                                        onChange={(e) => setData('address', e.target.value)}
                                        className="w-full p-2 mt-1 border rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder={data.network === 'lightning' ? 'Paste your Lightning invoice here' : data.network === 'usdt' ? 'Paste your USDT address here' : 'Paste your Bitcoin address here'}
                                        rows={3}
                                    />
                                    {formErrors.address && <div className="text-red-500 text-sm">{formErrors.address}</div>}
                                    {errors.address && <div className="text-red-500 text-sm">{errors.address}</div>}
                                </div>
                                
                                <div className="flex space-x-4 mt-6">
                                    <button
                                        onClick={() => setShowNetworkOptions(false)}
                                        className="px-4 py-2 bg-gray-300 text-gray-800 rounded-md hover:bg-gray-400 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600"
                                    >
                                        Back
                                    </button>
                                    <button
                                        onClick={handleSubmit}
                                        disabled={isProcessing}
                                        className={`px-4 py-2 bg-[#FF6F00] text-white rounded-md hover:bg-[#960018] ${
                                            isProcessing ? 'opacity-50 cursor-not-allowed' : ''
                                        }`}
                                    >
                                        {isProcessing ? 'Processing...' : 'Proceed with Withdrawal'}
                                    </button>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
} 