import { Head } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { useEffect, useState } from 'react';
import { router } from '@inertiajs/react';
import { toast } from 'react-hot-toast';

export default function WalletIndex({ balance, GGPayoutBalance }) {
    const [isRefreshing, setIsRefreshing] = useState(false);

    const handleRefresh = () => {
        setIsRefreshing(true);
        router.reload({ 
            only: ['balance', 'GGPayoutBalance'],
            onFinish: () => setIsRefreshing(false)
        });
    };

    const formatBalance = (amount, currency) => {
        try {
            if (currency === 'SATS') {
                // Format SATS with commas and no decimal places
                return new Intl.NumberFormat('en-US', {
                    style: 'decimal',
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 0
                }).format(amount) + ' sats';
            }

            if (currency === 'USDT') {
                return new Intl.NumberFormat('en-US', {
                    style: 'decimal',
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                }).format(amount/100) + ' USDT';
            }

            // For other currencies, use a simple number format if currency code is invalid
            try {
                return new Intl.NumberFormat('en-US', {
                    style: 'currency',
                    currency: currency,
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 8
                }).format(amount);
            } catch (e) {
                // Fallback to simple number format if currency code is invalid
                return new Intl.NumberFormat('en-US', {
                    style: 'decimal',
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 8
                }).format(amount) + ' ' + currency;
            }
        } catch (e) {
            console.error('Error formatting balance:', e);
            return `${amount} ${currency}`;
        }
    };

    const getCurrencyLabel = (currency) => {
        switch (currency) {
            case 'SATS':
                return 'Bitcoin (Sats)';
            default:
                return currency.toUpperCase();
        }
    };

    // Filter out any non-currency entries from the balance object
    const validBalances = balance ? Object.entries(balance).filter(([currency]) => 
        typeof currency === 'string' && currency.length > 0
    ) : [];

    return (
        <AdminLayout>
            <Head title="Wallet Balance" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* TrySpeed Balance Section */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <div className="flex justify-between items-center mb-6">
                                <h2 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                                    TrySpeed Merchant Balance
                                </h2>
                                <button
                                    onClick={handleRefresh}
                                    disabled={isRefreshing}
                                    className="inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700 focus:bg-indigo-700 active:bg-indigo-900 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition ease-in-out duration-150 disabled:opacity-50"
                                >
                                    {isRefreshing ? 'Refreshing...' : 'Refresh Balance'}
                                </button>
                            </div>

                            {validBalances.length > 0 ? (
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                    {validBalances.map(([currency, amount]) => (
                                        <div
                                            key={currency}
                                            className="bg-gray-50 dark:bg-gray-700 rounded-lg p-6 shadow-sm"
                                        >
                                            <div className="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">
                                                {getCurrencyLabel(currency)}
                                            </div>
                                            <div className="text-2xl font-bold text-gray-900 dark:text-white">
                                                {formatBalance(amount, currency)}
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            ) : (
                                <p className="text-gray-500 dark:text-gray-400">No balance information available</p>
                            )}
                        </div>
                    </div>

                    {/* GGPayout Balance Section */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex justify-between items-center mb-6">
                                <h2 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                                    GGPayout Merchant Balance
                                </h2>
                                <button
                                    onClick={handleRefresh}
                                    disabled={isRefreshing}
                                    className="inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700 focus:bg-indigo-700 active:bg-indigo-900 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition ease-in-out duration-150 disabled:opacity-50"
                                >
                                    {isRefreshing ? 'Refreshing...' : 'Refresh Balance'}
                                </button>
                            </div>
                            {GGPayoutBalance && Object.entries(GGPayoutBalance).length > 0 ? (
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                    {Object.entries(GGPayoutBalance).map(([currency, details]) => (
                                        <div
                                            key={currency}
                                            className="bg-gray-50 dark:bg-gray-700 rounded-lg p-6 shadow-sm"
                                        >
                                            <div className="text-sm font-medium text-gray-500 dark:text-gray-400 mb-3">
                                                {getCurrencyLabel(currency)}
                                            </div>
                                            <div className="space-y-2">
                                                <div>
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">Balance: </span>
                                                    <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                        {formatBalance(details.balance, currency)}
                                                    </span>
                                                </div>
                                                <div>
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">Available: </span>
                                                    <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                        {formatBalance(details.availableBalance, currency)}
                                                    </span>
                                                </div>
                                                <div>
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">Pending: </span>
                                                    <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                        {formatBalance(details.transferPendingAmount, currency)}
                                                    </span>
                                                </div>
                                                <div>
                                                    <span className="text-sm text-gray-500 dark:text-gray-400">Delayed: </span>
                                                    <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                        {formatBalance(details.delayAmount, currency)}
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            ) : (
                                <p className="text-gray-500 dark:text-gray-400">No GGPayout balance information available</p>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}