import { useState, useCallback } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link, router } from '@inertiajs/react';
import Pagination from '@/Components/Pagination';
import { toast } from 'react-hot-toast';
import { usePage } from '@inertiajs/react';

const debounce = (func, wait) => {
    let timeout;
    return (...args) => {
        clearTimeout(timeout);
        timeout = setTimeout(() => func(...args), wait);
    };
};

export default function Users({ users, filters }) {
    const { auth } = usePage().props;
    const [searchParams, setSearchParams] = useState({
        search: filters.search || '',
        role: filters.role || '',
        status: filters.status || ''
    });

    const debouncedSearch = useCallback(
        debounce((params) => {
            router.get(route('admin.users.index'), params, {
                preserveState: true,
                preserveScroll: true,
                replace: true
            });
        }, 300),
        []
    );

    const handleFilterChange = useCallback((e) => {
        const { name, value } = e.target;
        setSearchParams(prev => {
            const newParams = { ...prev, [name]: value };
            debouncedSearch(newParams);
            return newParams;
        });
    }, [debouncedSearch]);

    const clearFilters = () => {
        const clearedParams = {
            search: '',
            role: '',
            status: ''
        };
        setSearchParams(clearedParams);
        router.get(route('admin.users.index'), clearedParams, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleDelete = (userId) => {
        if (confirm('Are you sure you want to delete this user?')) {
            router.delete(route('admin.users.destroy', userId), {
                onSuccess: () => {
                    toast.success('User deleted successfully');
                },
                onError: (errors) => {
                    toast.error(errors.error || 'Failed to delete user');
                }
            });
        }
    };

    const handleToggleStatus = (userId) => {
        router.patch(route('admin.users.toggle-status', userId), {}, {
            onSuccess: () => {
                toast.success('User status updated');
            },
            onError: (errors) => {
                toast.error(errors.error || 'Failed to update user status');
            }
        });
    };

    return (
        <AdminLayout>
            <Head title="User Management" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header Section */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] p-4 sm:p-6 rounded-[15px] shadow-lg mb-6">
                        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 sm:gap-0">
                            <div>
                                <h1 className="text-2xl sm:text-3xl font-bold text-white">User Management</h1>
                                <p className="text-sm sm:text-base text-gray-100 dark:text-gray-200 mt-2">Manage system users and their roles</p>
                            </div>
                            <Link
                                href={route('admin.users.create')}
                                className="w-full sm:w-auto bg-white text-[#d34a40] px-4 sm:px-6 py-2 sm:py-3 rounded-lg hover:bg-gray-100 transition-colors flex items-center justify-center sm:justify-start shadow-md text-sm sm:text-base"
                            >
                                <i className="fas fa-plus mr-2"></i>
                                Add New User
                            </Link>
                        </div>
                    </div>

                    {/* Filters Section */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6 mb-6">
                        <div className="flex flex-wrap gap-4 items-end">
                            <div className="flex-1 min-w-[200px]">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Search
                                </label>
                                <input
                                    type="text"
                                    name="search"
                                    value={searchParams.search}
                                    onChange={handleFilterChange}
                                    placeholder="Search by name or IDN..."
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                />
                            </div>
                            <div className="w-[200px]">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Role
                                </label>
                                <select
                                    name="role"
                                    value={searchParams.role}
                                    onChange={handleFilterChange}
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                >
                                    <option value="">All Roles</option>
                                    <option value="customer">Customer</option>
                                    <option value="cashier">Cashier</option>
                                    <option value="admin">Admin</option>
                                </select>
                            </div>
                            <div className="w-[200px]">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Status
                                </label>
                                <select
                                    name="status"
                                    value={searchParams.status}
                                    onChange={handleFilterChange}
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                >
                                    <option value="">All Statuses</option>
                                    <option value="active">Active</option>
                                    <option value="banned">Banned</option>
                                </select>
                            </div>
                            <button
                                onClick={clearFilters}
                                className="px-4 py-2 bg-gray-100 text-gray-600 rounded-lg hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600"
                            >
                                Clear Filters
                            </button>
                        </div>
                    </div>

                    {/* Users Table */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg overflow-hidden">
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] text-white">
                                    <tr>
                                        <th className="px-6 py-4 text-left">Name</th>
                                        <th className="px-6 py-4 text-left">IDN No</th>
                                        <th className="px-6 py-4 text-left">Role</th>
                                        <th className="px-6 py-4 text-left">Status</th>
                                        <th className="px-6 py-4 text-left">Created At</th>
                                        <th className="px-6 py-4 text-left">Actions</th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                    {users.data.map(user => (
                                        <tr key={user.id} className="hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                            <td className="px-6 py-4 text-gray-900 dark:text-gray-100">{user.name}</td>
                                            <td className="px-6 py-4 text-gray-900 dark:text-gray-100">{user.idn_no}</td>
                                            <td className="px-6 py-4">
                                                <span className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                    user.role === 'admin' 
                                                        ? 'bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400'
                                                        : user.role === 'cashier'
                                                        ? 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400'
                                                        : 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400'
                                                }`}>
                                                    {user.role.charAt(0).toUpperCase() + user.role.slice(1)}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4">
                                                <span className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                    user.status === 'active'
                                                        ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400'
                                                        : 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                                }`}>
                                                    {user.status.charAt(0).toUpperCase() + user.status.slice(1)}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4 text-gray-900 dark:text-gray-100">
                                                {new Date(user.created_at).toLocaleDateString()}
                                            </td>
                                            <td className="px-6 py-4">
                                                <div className="flex space-x-3">
                                                    <Link
                                                        href={route('admin.users.edit', user.id)}
                                                        className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 transition-colors"
                                                        title="Edit User"
                                                    >
                                                        <i className="fas fa-edit"></i>
                                                    </Link>
                                                    {user.role === 'customer' && (
                                                        <>
                                                            <Link
                                                                href={route('admin.customer-transaction-history.view', user.id)}
                                                                className="text-green-600 dark:text-green-400 hover:text-green-800 dark:hover:text-green-300 transition-colors p-1 rounded hover:bg-green-50 dark:hover:bg-green-900/20"
                                                                title={`View ${user.name}'s Transaction History`}
                                                            >
                                                                <i className="fas fa-history text-lg"></i>
                                                            </Link>
                                                            <a
                                                                href={route('admin.customer-transaction-history.download', user.id)}
                                                                className="text-purple-600 dark:text-purple-400 hover:text-purple-800 dark:hover:text-purple-300 transition-colors p-1 rounded hover:bg-purple-50 dark:hover:bg-purple-900/20"
                                                                title={`Download ${user.name}'s Transaction Report`}
                                                                target="_blank"
                                                                rel="noopener noreferrer"
                                                            >
                                                                <i className="fas fa-download text-lg"></i>
                                                            </a>
                                                        </>
                                                    )}
                                                    {user.id !== auth.user.id && (
                                                        <>
                                                            {/* Freeze/Ban Toggle */}
                                                            <button
                                                                onClick={() => handleToggleStatus(user.id)}
                                                                className={user.status === 'banned' ? 'text-yellow-600 dark:text-yellow-400 hover:text-yellow-800 dark:hover:text-yellow-300 transition-colors' : 'text-gray-500 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-300 transition-colors'}
                                                                title={user.status === 'banned' ? 'Unban User' : 'Freeze/Ban User'}
                                                            >
                                                                {user.status === 'banned' ? (
                                                                    <i className="fas fa-ban"></i>
                                                                ) : (
                                                                    <i className="fas fa-snowflake"></i>
                                                                )}
                                                            </button>
                                                            {/* Delete User */}
                                                            <button
                                                                onClick={() => handleDelete(user.id)}
                                                                className="text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300 transition-colors"
                                                                title="Delete User"
                                                            >
                                                                <i className="fas fa-trash"></i>
                                                            </button>
                                                        </>
                                                    )}
                                                </div>
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                        <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                            <Pagination links={users.links} />
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 