import { useForm } from '@inertiajs/react';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';
import { toast } from 'react-hot-toast';
import { useEffect } from 'react';

export default function UserForm({ user = null, isEditing = false, idn_no = null }) {
    const { data, setData, post, put, processing, errors, reset } = useForm({
        name: user?.name || '',
        idn_no: user?.idn_no || idn_no,
        password: '',
        role: user?.role || 'customer',
        status: user?.status || 'active',
    });

    useEffect(() => {
        if (idn_no) {
            setData('idn_no', idn_no);
        }
    }, [idn_no]);

    const handleSubmit = (e) => {
        e.preventDefault();

        if (isEditing) {
            put(route('admin.users.update', user.id), {
                onSuccess: () => {
                    toast.success('User updated successfully');
                    reset();
                },
                onError: (errors) => {
                    toast.error('Please correct the errors in the form');
                }
            });
        } else {
            post(route('admin.users.store'), {
                onSuccess: () => {
                    toast.success('User created successfully');
                    reset();
                },
                onError: (errors) => {
                    toast.error('Please correct the errors in the form');
                }
            });
        }
    };

    const handleCopyDetails = () => {
        const details = `Name: ${data.name}\nIDN No: ${data.idn_no}\nRole: ${data.role}\nPassword: ${data.password}`;
        navigator.clipboard.writeText(details).then(() => {
            toast.success('User details copied to clipboard');
        }).catch(() => {
            toast.error('Failed to copy details');
        });
    };

    return (
        <form onSubmit={handleSubmit} className="space-y-6">
            <div>
                <InputLabel htmlFor="name" value="Name" />
                <TextInput
                    id="name"
                    type="text"
                    name="name"
                    value={data.name}
                    className="mt-1 block w-full"
                    onChange={(e) => setData('name', e.target.value)}
                    required
                />
                <InputError message={errors.name} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="idn_no" value="IDN No" />
                <TextInput
                    id="idn_no"
                    type="text"
                    name="idn_no"
                    readOnly
                    value={idn_no || data.idn_no}
                    className="mt-1 block w-full"
                    onChange={(e) => setData('idn_no', e.target.value)}
                    required
                />
                <InputError message={errors.idn_no} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="password" value={isEditing ? "New Password (leave blank to keep current)" : "Password"} />
                <TextInput
                    id="password"
                    type="password"
                    name="password"
                    value={data.password}
                    className="mt-1 block w-full"
                    onChange={(e) => setData('password', e.target.value)}
                    required={!isEditing}
                />
                <InputError message={errors.password} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="role" value="Role" />
                <select
                    id="role"
                    name="role"
                    value={data.role}
                    onChange={(e) => setData('role', e.target.value)}
                    className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40] rounded-md shadow-sm"
                    required
                >
                    <option value="customer">Customer</option>
                    <option value="cashier">Cashier</option>
                    <option value="admin">Admin</option>
                </select>
                <InputError message={errors.role} className="mt-2" />
            </div>

            {isEditing && (
                <div>
                    <InputLabel htmlFor="status" value="Status" />
                    <select
                        id="status"
                        name="status"
                        value={data.status}
                        onChange={(e) => setData('status', e.target.value)}
                        className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40] rounded-md shadow-sm"
                        required
                    >
                        <option value="active">Active</option>
                        <option value="banned">Banned</option>
                    </select>
                    <InputError message={errors.status} className="mt-2" />
                </div>
            )}

            <div className="flex items-center justify-end gap-4">
                <PrimaryButton type="button" onClick={handleCopyDetails}>
                    Copy Details
                </PrimaryButton>
                <PrimaryButton disabled={processing}>
                    {isEditing ? 'Update User' : 'Create User'}
                </PrimaryButton>
            </div>
        </form>
    );
}
