import { useState, useEffect, useCallback, Suspense } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link, router } from '@inertiajs/react';
import Pagination from '@/Components/Pagination';

// Optimize debounce with useCallback
const debounce = (func, wait) => {
    let timeout;
    return (...args) => {
        clearTimeout(timeout);
        timeout = setTimeout(() => func(...args), wait);
    };
};

export default function Transactions({ transactions, filters }) {
    const [searchParams, setSearchParams] = useState({
        search: filters.search || '',
        type: filters.type || '',
        status: filters.status || ''
    });

    const debouncedSearch = useCallback(
        debounce((params) => {
            router.get(route('admin.transactions.index'), params, {
                preserveState: true,
                preserveScroll: true,
                replace: true
            });
        }, 300),
        []
    );

    // Memoize the filter change handler
    const handleFilterChange = useCallback((e) => {
        const { name, value } = e.target;
        setSearchParams(prev => {
            const newParams = { ...prev, [name]: value };
            debouncedSearch(newParams);
            return newParams;
        });
    }, [debouncedSearch]);

    // Clear filters
    const clearFilters = () => {
        const clearedParams = {
            search: '',
            type: '',
            status: ''
        };
        setSearchParams(clearedParams);
        router.get(route('admin.transactions.index'), clearedParams, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const getStatusColor = (status) => {
        const colors = {
            unpaid: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400',
            paid: 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400',
            expired: 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400',
            cancelled: 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400'
        };
        return colors[status] || 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400';
    };

    return (
        <AdminLayout>
            <Head title="Transactions">
                {/* Add preload for critical resources */}
                <link rel="preload" href="/fonts/fa-solid-900.woff2" as="font" type="font/woff2" crossOrigin="anonymous" />
            </Head>
            
            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header Section */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] p-4 sm:p-6 rounded-[15px] shadow-lg mb-4 sm:mb-6">
                        <h1 className="text-2xl sm:text-3xl font-bold text-white">Transactions</h1>
                        <p className="text-sm sm:text-base text-gray-100 dark:text-gray-200 mt-1 sm:mt-2">View and manage all transactions</p>
                    </div>

                    {/* Wrap table in Suspense for lazy loading */}
                    <Suspense fallback={<div>Loading...</div>}>
                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-lg rounded-[15px] border border-gray-200 dark:border-gray-700">
                            <div className="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4 lg:gap-0 mb-6 p-4 lg:p-6">
                                <h2 className="text-xl lg:text-2xl font-semibold dark:text-white">Transactions</h2>
                                <div className="flex flex-col sm:flex-row gap-3 lg:gap-4 items-stretch sm:items-center w-full lg:w-auto">
                                    <input
                                        type="text"
                                        name="search"
                                        placeholder="Search transactions..."
                                        value={searchParams.search}
                                        onChange={handleFilterChange}
                                        className="w-full sm:w-auto rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 text-sm"
                                    />
                                    <select
                                        name="type"
                                        value={searchParams.type}
                                        onChange={handleFilterChange}
                                        className="w-full sm:w-auto rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 text-sm"
                                    >
                                        <option value="">All Types</option>
                                        <option value="deposit">Deposits</option>
                                        <option value="withdrawal">Withdrawals</option>
                                    </select>
                                    <select
                                        name="status"
                                        value={searchParams.status}
                                        onChange={handleFilterChange}
                                        className="w-full sm:w-auto rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 text-sm"
                                    >
                                        <option value="">All Statuses</option>
                                        <option value="unpaid">Unpaid</option>
                                        <option value="paid">Paid</option>
                                        <option value="expired">Expired</option>
                                        <option value="cancelled">Cancelled</option>
                                    </select>
                                    <button
                                        onClick={clearFilters}
                                        className="w-full sm:w-auto px-3 py-2 text-sm bg-gray-100 text-gray-600 rounded-md hover:bg-gray-200 text-center"
                                    >
                                        Clear Filters
                                    </button>
                                </div>
                            </div>

                            <div className="overflow-x-auto">
                                <table className="w-full">
                                    <thead className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] text-white">
                                        <tr>
                                            <th className="px-6 py-4 text-left">Invoice ID</th>
                                            <th className="px-6 py-4 text-left">Transaction ID</th>
                                            <th className="px-6 py-4 text-left">Username</th>
                                            <th className="px-6 py-4 text-left">Amount</th>
                                            <th className="px-6 py-4 text-left">Type</th>
                                            <th className="px-6 py-4 text-left">Status</th>
                                            <th className="px-6 py-4 text-left">Date</th>
                                            <th className="px-6 py-4 text-left">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                        {transactions.data.map((transaction) => (
                                            <tr key={transaction.id} className="hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                                <td className="px-6 py-4 whitespace-nowrap text-gray-900 dark:text-gray-100">{transaction.invoice_id}</td>
                                                <td className="px-6 py-4 whitespace-nowrap text-gray-900 dark:text-gray-100">{transaction.transaction_id}</td>
                                                <td className="px-6 py-4 whitespace-nowrap text-gray-900 dark:text-gray-100">{transaction.username}</td>
                                                <td className="px-6 py-4 whitespace-nowrap text-gray-900 dark:text-gray-100">
                                                    {transaction.amount} {transaction.currency}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-gray-900 dark:text-gray-100 capitalize">{transaction.type}</td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span className={`px-3 py-1 rounded-full text-sm font-medium ${getStatusColor(transaction.payment_status)}`}>
                                                        {transaction.payment_status}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-gray-900 dark:text-gray-100">
                                                    {new Date(transaction.created_at).toLocaleDateString()}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <Link
                                                        href={route('admin.transactions.show', transaction.id)}
                                                        className="text-[#d34a40] dark:text-[#ff6b61] hover:text-[#b03a30] dark:hover:text-[#ff8a82] transition-colors"
                                                    >
                                                        <i className="fas fa-eye"></i>
                                                    </Link>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>

                            <div className="p-6 border-t border-gray-200 dark:border-gray-700">
                                <Pagination links={transactions.links} />
                            </div>
                        </div>
                    </Suspense>
                </div>
            </div>
        </AdminLayout>
    );
} 