import AdminLayout from '@/Layouts/AdminLayout';
import { Head } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import axios from 'axios';

const TransactionStats = () => {
    const [stats, setStats] = useState(null);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const [period, setPeriod] = useState('daily');
    const [startDate, setStartDate] = useState('');
    const [endDate, setEndDate] = useState('');
    const [activeTab, setActiveTab] = useState('overview');
    const [comparisonData, setComparisonData] = useState(null);
    const [comparisonLoading, setComparisonLoading] = useState(false);
    const [currentComparePeriod, setCurrentComparePeriod] = useState('daily');
    const [comparePeriod, setComparePeriod] = useState('daily');

    const fetchStats = async () => {
        setLoading(true);
        setError(null);
        try {
            const params = { period };
            if (period === 'custom') {
                params.start_date = startDate;
                params.end_date = endDate;
            }

            const response = await axios.get('/admin/transaction-stats/api', { params });
            setStats(response.data);
        } catch (error) {
            console.error('Error fetching stats:', error);
            setError('Failed to load transaction statistics. Please try again.');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchStats();
    }, [period, startDate, endDate]);

    const fetchComparison = async () => {
        setComparisonLoading(true);
        try {
            const params = {
                current_period: currentComparePeriod,
                compare_period: currentComparePeriod // Use same period type for comparison
            };

            // Add custom period parameters if needed
            if (currentComparePeriod === 'custom') {
                params.start_date = startDate;
                params.end_date = endDate;
                params.compare_start_date = compareStartDate;
                params.compare_end_date = compareEndDate;
            }

            const response = await axios.get('/admin/transaction-stats/compare', { params });
            setComparisonData(response.data);
        } catch (error) {
            console.error('Error fetching comparison:', error);
        } finally {
            setComparisonLoading(false);
        }
    };

    useEffect(() => {
        if (activeTab === 'comparison') {
            fetchComparison();
        }
    }, [activeTab, currentComparePeriod, comparePeriod]);

    const paymentMethodColors = {
        bitcoin: 'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400',
        usdt: 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400',
        payken: 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400',
        cashapp: 'bg-emerald-100 text-emerald-800 dark:bg-emerald-900/30 dark:text-emerald-400',
        ecashapp: 'bg-teal-100 text-teal-800 dark:bg-teal-900/30 dark:text-teal-400',
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount);
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'success': return 'text-green-600 dark:text-green-400';
            case 'warning': return 'text-yellow-600 dark:text-yellow-400';
            case 'danger': return 'text-red-600 dark:text-red-400';
            default: return 'text-gray-600 dark:text-gray-400';
        }
    };

    const renderOverviewTab = () => (
        <div className="space-y-6">
            {/* Summary Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Total Processed</h3>
                        <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-full">
                            <i className="fas fa-dollar-sign text-green-600 dark:text-green-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-gray-900 dark:text-white">
                        {stats.totals.formatted_total_processed}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        {stats.totals.total_transactions} transactions
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Total Deposits</h3>
                        <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-full">
                            <i className="fas fa-arrow-up text-green-600 dark:text-green-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                        {stats.totals.formatted_total_deposits}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        All payment methods
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Total Withdrawals</h3>
                        <div className="p-2 bg-red-100 dark:bg-red-900/30 rounded-full">
                            <i className="fas fa-arrow-down text-red-600 dark:text-red-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-red-600 dark:text-red-400">
                        {stats.totals.formatted_total_withdrawals}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        All payment methods
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Success Rate</h3>
                        <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                            <i className="fas fa-chart-line text-blue-600 dark:text-blue-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-gray-900 dark:text-white">
                        {stats.totals.success_rate}%
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        {stats.totals.pending_transactions} pending, {stats.totals.failed_transactions} failed
                    </p>
                </div>
            </div>

            {/* Additional Stats Row */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Net Revenue</h3>
                        <div className="p-2 bg-emerald-100 dark:bg-emerald-900/30 rounded-full">
                            <i className="fas fa-chart-line text-emerald-600 dark:text-emerald-400"></i>
                        </div>
                    </div>
                    <p className={`text-2xl font-bold ${stats.totals.net_revenue >= 0 ? 'text-emerald-600 dark:text-emerald-400' : 'text-red-600 dark:text-red-400'}`}>
                        {stats.totals.formatted_net_revenue}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        Deposits - Withdrawals
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Pending Amount</h3>
                        <div className="p-2 bg-yellow-100 dark:bg-yellow-900/30 rounded-full">
                            <i className="fas fa-clock text-yellow-600 dark:text-yellow-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                        {stats.totals.formatted_pending_amount}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        {stats.totals.pending_transactions} transactions
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Failed Amount</h3>
                        <div className="p-2 bg-red-100 dark:bg-red-900/30 rounded-full">
                            <i className="fas fa-times-circle text-red-600 dark:text-red-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-red-600 dark:text-red-400">
                        {stats.totals.formatted_failed_amount}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        {stats.totals.failed_transactions} transactions
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Avg Transaction</h3>
                        <div className="p-2 bg-purple-100 dark:bg-purple-900/30 rounded-full">
                            <i className="fas fa-calculator text-purple-600 dark:text-purple-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-gray-900 dark:text-white">
                        {stats.totals.formatted_avg_amount}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        Per transaction
                    </p>
                </div>
            </div>

            {/* Period and Methods Row */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Period</h3>
                        <div className="p-2 bg-indigo-100 dark:bg-indigo-900/30 rounded-full">
                            <i className="fas fa-calendar text-indigo-600 dark:text-indigo-400"></i>
                        </div>
                    </div>
                    <p className="text-lg font-semibold text-gray-900 dark:text-white capitalize">
                        {stats.summary.period_label}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        {new Date(stats.start_date).toLocaleDateString()} - {new Date(stats.end_date).toLocaleDateString()}
                    </p>
                </div>

                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="font-semibold text-gray-900 dark:text-white">Payment Methods</h3>
                        <div className="p-2 bg-yellow-100 dark:bg-yellow-900/30 rounded-full">
                            <i className="fas fa-credit-card text-yellow-600 dark:text-yellow-400"></i>
                        </div>
                    </div>
                    <p className="text-2xl font-bold text-gray-900 dark:text-white">
                        {stats.summary.total_payment_methods}
                    </p>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        Active methods
                    </p>
                </div>
            </div>

            {/* Payment Method Breakdown */}
            <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Payment Method Breakdown</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {Object.entries(stats.payment_methods).map(([method, data]) => (
                        <div key={method} className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 hover:shadow-md transition-shadow">
                            <div className="flex items-center justify-between mb-3">
                                <h3 className="font-semibold text-gray-900 dark:text-white capitalize">
                                    {method === 'ecashapp' ? 'Cash App' : method.toUpperCase()}
                                </h3>
                                <span className={`px-2 py-1 rounded-full text-xs font-medium ${paymentMethodColors[method] || 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300'}`}>
                                    {data.transaction_count} tx
                                </span>
                            </div>
                            <div className="space-y-2">
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Total:</span>
                                    <span className="font-semibold text-gray-900 dark:text-white">
                                        {formatCurrency(data.total_amount)}
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Deposits:</span>
                                    <span className="text-green-600 dark:text-green-400 font-medium">
                                        {formatCurrency(data.deposits)}
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Withdrawals:</span>
                                    <span className="text-red-600 dark:text-red-400 font-medium">
                                        {formatCurrency(data.withdrawals)}
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Net Revenue:</span>
                                    <span className={`font-medium ${data.net_revenue >= 0 ? 'text-emerald-600 dark:text-emerald-400' : 'text-red-600 dark:text-red-400'}`}>
                                        {formatCurrency(data.net_revenue)}
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Pending:</span>
                                    <span className="text-yellow-600 dark:text-yellow-400 font-medium">
                                        {formatCurrency(data.pending_amount)}
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Failed:</span>
                                    <span className="text-red-600 dark:text-red-400 font-medium">
                                        {formatCurrency(data.failed_amount)}
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Success Rate:</span>
                                    <span className={`font-medium ${getStatusColor(data.success_rate >= 80 ? 'success' : data.success_rate >= 60 ? 'warning' : 'danger')}`}>
                                        {data.success_rate}%
                                    </span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-sm text-gray-600 dark:text-gray-400">Avg Amount:</span>
                                    <span className="font-medium text-gray-900 dark:text-white">
                                        {formatCurrency(data.avg_amount)}
                                    </span>
                                </div>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        </div>
    );

    const renderTopUsersTab = () => (
        <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
            <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Top Users by Transaction Volume</h2>
            <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead className="bg-gray-50 dark:bg-gray-700">
                        <tr>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Rank
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Username
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Total Amount
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Transactions
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Avg Amount
                            </th>
                        </tr>
                    </thead>
                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                        {stats.top_users.map((user, index) => (
                            <tr key={user.user_id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                    #{index + 1}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                    {user.username || `User ${user.user_id}`}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900 dark:text-white">
                                    {formatCurrency(user.total_amount)}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                    {user.transaction_count}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                    {formatCurrency(user.total_amount / user.transaction_count)}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>
        </div>
    );

    const renderAnalyticsTab = () => (
        <div className="space-y-6">
            <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Transaction Summary</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div className="text-center">
                        <div className="text-3xl font-bold text-green-600 dark:text-green-400">
                            {stats.totals.formatted_total_deposits}
                        </div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">Total Deposits</div>
                    </div>
                    <div className="text-center">
                        <div className="text-3xl font-bold text-red-600 dark:text-red-400">
                            {stats.totals.formatted_total_withdrawals}
                        </div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">Total Withdrawals</div>
                    </div>
                    <div className="text-center">
                        <div className="text-3xl font-bold text-blue-600 dark:text-blue-400">
                            {stats.totals.total_transactions}
                        </div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">Total Transactions</div>
                    </div>
                </div>
            </div>

            <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Performance Metrics</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">Success Rate by Payment Method</h3>
                        <div className="space-y-3">
                            {Object.entries(stats.payment_methods).map(([method, data]) => (
                                <div key={method} className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300 capitalize">
                                        {method === 'ecashapp' ? 'Cash App' : method}
                                    </span>
                                    <div className="flex items-center space-x-2">
                                        <div className="w-24 bg-gray-200 dark:bg-gray-700 rounded-full h-2">
                                            <div 
                                                className={`h-2 rounded-full ${data.success_rate >= 80 ? 'bg-green-500' : data.success_rate >= 60 ? 'bg-yellow-500' : 'bg-red-500'}`}
                                                style={{ width: `${data.success_rate}%` }}
                                            ></div>
                                        </div>
                                        <span className="text-sm font-medium text-gray-900 dark:text-white w-12 text-right">
                                            {data.success_rate}%
                                        </span>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                    <div>
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">Transaction Status</h3>
                        <div className="space-y-3">
                            <div className="flex justify-between items-center">
                                <span className="text-sm text-gray-600 dark:text-gray-400">Successful</span>
                                <span className="text-sm font-medium text-green-600 dark:text-green-400">
                                    {stats.totals.total_transactions}
                                </span>
                            </div>
                            <div className="flex justify-between items-center">
                                <span className="text-sm text-gray-600 dark:text-gray-400">Pending</span>
                                <span className="text-sm font-medium text-yellow-600 dark:text-yellow-400">
                                    {stats.totals.pending_transactions}
                                </span>
                            </div>
                            <div className="flex justify-between items-center">
                                <span className="text-sm text-gray-600 dark:text-gray-400">Failed</span>
                                <span className="text-sm font-medium text-red-600 dark:text-red-400">
                                    {stats.totals.failed_transactions}
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );

    const renderComparisonTab = () => (
        <div className="space-y-6">
            {/* Period Selection */}
            <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Period Comparison</h2>
                <div className="mb-4 p-4 bg-blue-50 dark:bg-blue-900/30 rounded-lg">
                    <div className="flex items-center">
                        <i className="fas fa-info-circle text-blue-600 dark:text-blue-400 mr-2"></i>
                        <span className="text-sm text-blue-800 dark:text-blue-200">
                            Compare current period with the previous period of the same type
                        </span>
                    </div>
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Select Period Type
                    </label>
                    <select
                        value={currentComparePeriod}
                        onChange={(e) => setCurrentComparePeriod(e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                    >
                        <option value="daily">Daily (Today vs Yesterday)</option>
                        <option value="weekly">Weekly (This Week vs Last Week)</option>
                        <option value="monthly">Monthly (This Month vs Last Month)</option>
                    </select>
                </div>
            </div>

            {/* Loading State */}
            {comparisonLoading && (
                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-center">
                        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
                        <span className="ml-3 text-gray-600 dark:text-gray-400">Loading comparison data...</span>
                    </div>
                </div>
            )}

            {/* Comparison Results */}
            {comparisonData && !comparisonLoading && (
                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Growth Analysis</h2>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                        {Object.entries(comparisonData.growth_rates).map(([metric, rate]) => (
                            <div key={metric} className="text-center p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                                <div className={`text-2xl font-bold ${rate >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                    {rate >= 0 ? '+' : ''}{rate}%
                                </div>
                                <div className="text-sm text-gray-600 dark:text-gray-400 capitalize">
                                    {metric.replace(/_/g, ' ')}
                                </div>
                                <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                    vs Previous {comparisonData.current_period}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            )}

            {/* Detailed Comparison */}
            {comparisonData && (
                <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                    <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-6">Detailed Comparison</h2>
                    <div className="overflow-x-auto">
                        <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                            <thead className="bg-gray-50 dark:bg-gray-700">
                                <tr>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                        Metric
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                        Current {comparisonData.current_period}
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                        Previous {comparisonData.current_period}
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                        Change
                                    </th>
                                </tr>
                            </thead>
                            <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                <tr>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                        Total Deposits
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.current_stats.formatted_total_deposits}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.compare_stats.formatted_total_deposits}
                                    </td>
                                    <td className={`px-6 py-4 whitespace-nowrap text-sm font-medium ${comparisonData.growth_rates.total_deposits >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                        {comparisonData.growth_rates.total_deposits >= 0 ? '+' : ''}{comparisonData.growth_rates.total_deposits}%
                                    </td>
                                </tr>
                                <tr>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                        Total Withdrawals
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.current_stats.formatted_total_withdrawals}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.compare_stats.formatted_total_withdrawals}
                                    </td>
                                    <td className={`px-6 py-4 whitespace-nowrap text-sm font-medium ${comparisonData.growth_rates.total_withdrawals >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                        {comparisonData.growth_rates.total_withdrawals >= 0 ? '+' : ''}{comparisonData.growth_rates.total_withdrawals}%
                                    </td>
                                </tr>
                                <tr>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                        Net Revenue
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.current_stats.formatted_net_revenue}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.compare_stats.formatted_net_revenue}
                                    </td>
                                    <td className={`px-6 py-4 whitespace-nowrap text-sm font-medium ${comparisonData.growth_rates.net_revenue >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                        {comparisonData.growth_rates.net_revenue >= 0 ? '+' : ''}{comparisonData.growth_rates.net_revenue}%
                                    </td>
                                </tr>
                                <tr>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                        Total Transactions
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.current_stats.total_transactions}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        {comparisonData.compare_stats.total_transactions}
                                    </td>
                                    <td className={`px-6 py-4 whitespace-nowrap text-sm font-medium ${comparisonData.growth_rates.total_transactions >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                        {comparisonData.growth_rates.total_transactions >= 0 ? '+' : ''}{comparisonData.growth_rates.total_transactions}%
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            )}
        </div>
    );

    return (
        <AdminLayout>
            <Head title="Transaction Statistics" />
            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 space-y-6">
                    {/* Header */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] overflow-hidden shadow-lg rounded-[15px]">
                        <div className="p-6 text-white">
                            <h1 className="text-3xl font-bold mb-2">Transaction Statistics</h1>
                            <p className="text-gray-100 dark:text-gray-200">
                                Comprehensive analytics and insights for payment processing
                            </p>
                        </div>
                    </div>

                    {/* Period Selection */}
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between mb-4">
                            <h2 className="font-semibold text-gray-900 dark:text-white text-lg">Select Time Period</h2>
                            <button
                                onClick={fetchStats}
                                disabled={loading}
                                className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                            >
                                <i className={`fas fa-sync-alt mr-2 ${loading ? 'animate-spin' : ''}`}></i>
                                Refresh
                            </button>
                        </div>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            {['daily', 'weekly', 'monthly', 'custom'].map((p) => (
                                <button
                                    key={p}
                                    onClick={() => setPeriod(p)}
                                    className={`p-3 rounded-lg border transition-colors ${
                                        period === p
                                            ? 'bg-blue-500 text-white border-blue-500'
                                            : 'bg-gray-50 dark:bg-gray-700 text-gray-700 dark:text-gray-300 border-gray-300 dark:border-gray-600'
                                    }`}
                                >
                                    {p.charAt(0).toUpperCase() + p.slice(1)}
                                </button>
                            ))}
                        </div>

                        {period === 'custom' && (
                            <div className="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Start Date
                                    </label>
                                    <input
                                        type="date"
                                        value={startDate}
                                        onChange={(e) => setStartDate(e.target.value)}
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                    />
                                </div>
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        End Date
                                    </label>
                                    <input
                                        type="date"
                                        value={endDate}
                                        onChange={(e) => setEndDate(e.target.value)}
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                    />
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Error State */}
                    {error && (
                        <div className="bg-red-50 dark:bg-red-900/30 border border-red-200 dark:border-red-800 p-4 rounded-lg">
                            <div className="flex items-center">
                                <i className="fas fa-exclamation-triangle text-red-600 dark:text-red-400 mr-3"></i>
                                <span className="text-red-800 dark:text-red-200">{error}</span>
                            </div>
                        </div>
                    )}

                    {/* Loading State */}
                    {loading && (
                        <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-center">
                                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
                                <span className="ml-3 text-gray-600 dark:text-gray-400">Loading statistics...</span>
                            </div>
                        </div>
                    )}

                    {/* Statistics Display */}
                    {stats && !loading && (
                        <>
                            {/* Quick Stats Bar */}
                            <div className="bg-white dark:bg-gray-800 p-4 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-center">
                                    <div>
                                        <div className="text-lg font-bold text-green-600 dark:text-green-400">
                                            {stats.summary.total_payment_methods}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400">Payment Methods</div>
                                    </div>
                                    <div>
                                        <div className="text-lg font-bold text-blue-600 dark:text-blue-400">
                                            {stats.summary.date_range_days}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400">Days</div>
                                    </div>
                                    <div>
                                        <div className="text-lg font-bold text-purple-600 dark:text-purple-400">
                                            {stats.totals.total_transactions}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400">Transactions</div>
                                    </div>
                                    <div>
                                        <div className="text-lg font-bold text-orange-600 dark:text-orange-400">
                                            {stats.totals.success_rate}%
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400">Success Rate</div>
                                    </div>
                                </div>
                            </div>

                            {/* Tab Navigation */}
                            <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                                <div className="border-b border-gray-200 dark:border-gray-700">
                                    <nav className="-mb-px flex space-x-8">
                                        {[
                                            { id: 'overview', label: 'Overview', icon: 'fas fa-chart-pie' },
                                            { id: 'top-users', label: 'Top Users', icon: 'fas fa-users' },
                                            { id: 'analytics', label: 'Analytics', icon: 'fas fa-chart-bar' },
                                            { id: 'comparison', label: 'Comparison', icon: 'fas fa-chart-line' }
                                        ].map((tab) => (
                                            <button
                                                key={tab.id}
                                                onClick={() => setActiveTab(tab.id)}
                                                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                                    activeTab === tab.id
                                                        ? 'border-blue-500 text-blue-600 dark:text-blue-400'
                                                        : 'border-transparent text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300 hover:border-gray-300 dark:hover:border-gray-600'
                                                }`}
                                            >
                                                <i className={`${tab.icon} mr-2`}></i>
                                                {tab.label}
                                            </button>
                                        ))}
                                    </nav>
                                </div>
                            </div>

                            {/* Tab Content */}
                            <div className="mt-6">
                                {activeTab === 'overview' && renderOverviewTab()}
                                {activeTab === 'top-users' && renderTopUsersTab()}
                                {activeTab === 'analytics' && renderAnalyticsTab()}
                                {activeTab === 'comparison' && renderComparisonTab()}
                            </div>

                            {/* Export Section */}
                            <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                                <h2 className="font-semibold text-gray-900 dark:text-white text-lg mb-4">Export Options</h2>
                                <div className="flex flex-wrap gap-3">
                                    <button
                                        onClick={() => {
                                            const dataStr = JSON.stringify(stats, null, 2);
                                            const dataBlob = new Blob([dataStr], {type: 'application/json'});
                                            const url = URL.createObjectURL(dataBlob);
                                            const link = document.createElement('a');
                                            link.href = url;
                                            link.download = `transaction-stats-${period}-${new Date().toISOString().split('T')[0]}.json`;
                                            link.click();
                                        }}
                                        className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                                    >
                                        <i className="fas fa-download mr-2"></i>
                                        Export JSON
                                    </button>
                                    <button
                                        onClick={() => {
                                            const csvData = [
                                                ['Payment Method', 'Total Amount', 'Transactions', 'Deposits', 'Withdrawals', 'Success Rate', 'Avg Amount'],
                                                ...Object.entries(stats.payment_methods).map(([method, data]) => [
                                                    method.toUpperCase(),
                                                    data.formatted_total,
                                                    data.transaction_count,
                                                    data.formatted_deposits,
                                                    data.formatted_withdrawals,
                                                    `${data.success_rate}%`,
                                                    data.formatted_avg_amount
                                                ])
                                            ];
                                            const csvContent = csvData.map(row => row.join(',')).join('\n');
                                            const blob = new Blob([csvContent], { type: 'text/csv' });
                                            const url = URL.createObjectURL(blob);
                                            const link = document.createElement('a');
                                            link.href = url;
                                            link.download = `transaction-stats-${period}-${new Date().toISOString().split('T')[0]}.csv`;
                                            link.click();
                                        }}
                                        className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                                    >
                                        <i className="fas fa-file-csv mr-2"></i>
                                        Export CSV
                                    </button>
                                </div>
                            </div>
                        </>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
};

export default TransactionStats; 