import { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, useForm } from '@inertiajs/react';

export default function Settings({ settings }) {
    const { data, setData, post, processing, errors } = useForm({
        settings: [
            ...Object.values(settings.payment_limits || {}),
            ...Object.values(settings.currency || {}),
            ...Object.values(settings.payment_methods || {}),
            ...Object.values(settings.content || {})
        ]
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route('admin.settings.update'));
    };

    const renderInput = (setting, index) => {
        switch (setting.type) {
            case 'boolean':
                return (
                    <div className="relative inline-block w-10 mr-2 align-middle select-none">
                        <input
                            type="checkbox"
                            checked={setting.value === 'true'}
                            onChange={(e) => {
                                const newSettings = [...data.settings];
                                newSettings[index].value = e.target.checked.toString();
                                setData('settings', newSettings);
                            }}
                            className="toggle-checkbox absolute block w-6 h-6 rounded-full bg-white border-4 appearance-none cursor-pointer"
                        />
                        <label className="toggle-label block overflow-hidden h-6 rounded-full bg-gray-300 cursor-pointer"></label>
                    </div>
                );
            case 'array':
                return (
                    <div className="flex gap-2">
                        {JSON.parse(setting.value).map((currency, i) => (
                            <div key={i} className="flex items-center gap-2">
                                <input
                                    type="text"
                                    value={currency}
                                    onChange={(e) => {
                                        const values = JSON.parse(setting.value);
                                        values[i] = e.target.value;
                                        const newSettings = [...data.settings];
                                        newSettings[index].value = JSON.stringify(values);
                                        setData('settings', newSettings);
                                    }}
                                    className="mt-1 block w-24 rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 sm:text-sm"
                                />
                            </div>
                        ))}
                    </div>
                );
            case 'select':
                return (
                    <select
                        value={setting.value}
                        onChange={(e) => {
                            const newSettings = [...data.settings];
                            newSettings[index].value = e.target.value;
                            setData('settings', newSettings);
                        }}
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 sm:text-sm"
                    >
                        {JSON.parse(data.settings.find(s => s.key === 'available_currencies')?.value || '[]').map(currency => (
                            <option key={currency} value={currency}>{currency}</option>
                        ))}
                    </select>
                );
            case 'textarea':
                return (
                    <textarea
                        value={setting.value}
                        onChange={(e) => {
                            const newSettings = [...data.settings];
                            newSettings[index].value = e.target.value;
                            setData('settings', newSettings);
                        }}
                        rows={10}
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 sm:text-sm"
                    />
                );
            default:
                return (
                    <input
                        type={setting.type === 'number' ? 'number' : 'text'}
                        step={setting.type === 'number' ? '0.00001' : undefined}
                        value={setting.value}
                        onChange={(e) => {
                            const newSettings = [...data.settings];
                            newSettings[index].value = e.target.value;
                            setData('settings', newSettings);
                        }}
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 sm:text-sm"
                    />
                );
        }
    };

    const renderSettingsByGroup = (group, title) => (
        <div className="mb-8">
            <h2 className="text-lg font-semibold mb-6">{title}</h2>
            {data.settings
                .filter(setting => setting.group === group)
                .map((setting, index) => (
                    <div key={setting.key} className="mb-4">
                        <label className="block text-sm font-medium text-gray-700">
                            {setting.label}
                        </label>
                        {renderInput(setting, data.settings.findIndex(s => s.key === setting.key))}
                        {setting.description && (
                            <p className="mt-1 text-sm text-gray-500">{setting.description}</p>
                        )}
                    </div>
                ))}
        </div>
    );

    return (
        <AdminLayout>
            <Head title="Settings" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <form onSubmit={handleSubmit}>
                                {renderSettingsByGroup('payment_limits', 'Payment Limits')}
                                {renderSettingsByGroup('currency', 'Currency Settings')}
                                {renderSettingsByGroup('payment_methods', 'Payment Methods')}
                                {renderSettingsByGroup('content', 'Content Management')}

                                <div className="mt-6">
                                    <button
                                        type="submit"
                                        disabled={processing}
                                        className="inline-flex justify-center rounded-md border border-transparent bg-indigo-600 py-2 px-4 text-sm font-medium text-white shadow-sm hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2"
                                    >
                                        {processing ? 'Saving...' : 'Save Changes'}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
                    <style jsx>{`
                .toggle-checkbox:checked {
                    right: 0;
                    border-color: #68D391;
                }
                .toggle-checkbox:checked + .toggle-label {
                    background-color: #68D391;
                }
                .toggle-label {
                    transition: background-color 0.2s ease-in;
                }
            `}</style>
        </AdminLayout>
    );
}