import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link, router } from '@inertiajs/react';
import { useState, useEffect, useCallback } from 'react';
import axios from 'axios';
import toast from 'react-hot-toast';

const LogsIndex = ({ logs, filters, sources, users, stats }) => {
    const [localFilters, setLocalFilters] = useState(filters);
    const [searchValue, setSearchValue] = useState(filters.search || '');
    const [showClearModal, setShowClearModal] = useState(false);
    const [showExportModal, setShowExportModal] = useState(false);
    const [clearForm, setClearForm] = useState({ days: 30, type: 'all' });
    const [exportForm, setExportForm] = useState({ format: 'csv', type: 'all', date_from: '', date_to: '' });

    const updateFilters = (newFilters) => {
        setLocalFilters({ ...localFilters, ...newFilters });
        router.get(route('admin.logs.index'), { ...localFilters, ...newFilters }, {
            preserveState: true,
            replace: true,
        });
    };

    // Debounced search function
    const debouncedSearch = useCallback(
        (() => {
            let timeoutId;
            return (value) => {
                clearTimeout(timeoutId);
                timeoutId = setTimeout(() => {
                    updateFilters({ search: value });
                }, 500); // 500ms delay
            };
        })(),
        []
    );

    // Handle search input change
    const handleSearchChange = (e) => {
        const value = e.target.value;
        setSearchValue(value);
        debouncedSearch(value);
    };

    const handleClearLogs = async () => {
        try {
            const response = await axios.post(route('admin.logs.clear'), clearForm);
            toast.success(response.data.message);
            setShowClearModal(false);
            router.reload();
        } catch (error) {
            toast.error('Failed to clear logs');
        }
    };

    const handleExportLogs = async () => {
        try {
            const response = await axios.post(route('admin.logs.export'), exportForm, {
                responseType: 'blob'
            });
            
            const url = window.URL.createObjectURL(new Blob([response.data]));
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', `logs-${new Date().toISOString().split('T')[0]}.${exportForm.format}`);
            document.body.appendChild(link);
            link.click();
            link.remove();
            
            setShowExportModal(false);
            toast.success('Logs exported successfully');
        } catch (error) {
            toast.error('Failed to export logs');
        }
    };

    const getLogTypeColor = (type) => {
        switch (type) {
            case 'error': return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            case 'warning': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400';
            case 'info': return 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400';
            default: return 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300';
        }
    };

    const truncateMessage = (message, length = 100) => {
        return message.length > length ? message.substring(0, length) + '...' : message;
    };

    return (
        <AdminLayout>
            <Head title="System Logs" />
            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 space-y-6">
                    {/* Header */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] overflow-hidden shadow-lg rounded-[15px]">
                        <div className="p-6 text-white">
                            <h1 className="text-3xl font-bold mb-2">System Logs</h1>
                            <p className="text-gray-100 dark:text-gray-200">
                                Monitor and manage system activity logs
                            </p>
                        </div>
                    </div>

                    {/* Statistics Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                        <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="font-semibold text-gray-900 dark:text-white">Total Logs</h3>
                                <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                                    <i className="fas fa-list text-blue-600 dark:text-blue-400"></i>
                                </div>
                            </div>
                            <p className="text-2xl font-bold text-gray-900 dark:text-white">{stats.total_logs}</p>
                        </div>

                        <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="font-semibold text-gray-900 dark:text-white">Errors</h3>
                                <div className="p-2 bg-red-100 dark:bg-red-900/30 rounded-full">
                                    <i className="fas fa-exclamation-triangle text-red-600 dark:text-red-400"></i>
                                </div>
                            </div>
                            <p className="text-2xl font-bold text-red-600 dark:text-red-400">{stats.error_logs}</p>
                        </div>

                        <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="font-semibold text-gray-900 dark:text-white">Today</h3>
                                <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-full">
                                    <i className="fas fa-calendar-day text-green-600 dark:text-green-400"></i>
                                </div>
                            </div>
                            <p className="text-2xl font-bold text-gray-900 dark:text-white">{stats.today_logs}</p>
                        </div>

                        <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="font-semibold text-gray-900 dark:text-white">This Week</h3>
                                <div className="p-2 bg-purple-100 dark:bg-purple-900/30 rounded-full">
                                    <i className="fas fa-calendar-week text-purple-600 dark:text-purple-400"></i>
                                </div>
                            </div>
                            <p className="text-2xl font-bold text-gray-900 dark:text-white">{stats.this_week_logs}</p>
                        </div>
                    </div>

                    {/* Filters */}
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between mb-4">
                            <h2 className="font-semibold text-gray-900 dark:text-white text-lg">Filters</h2>
                            <div className="flex space-x-2">
                                <button
                                    onClick={() => setShowExportModal(true)}
                                    className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                                >
                                    <i className="fas fa-download mr-2"></i>
                                    Export
                                </button>
                                <button
                                    onClick={() => setShowClearModal(true)}
                                    className="flex items-center px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
                                >
                                    <i className="fas fa-trash mr-2"></i>
                                    Clear Old Logs
                                </button>
                            </div>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            {/* Search */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Search
                                </label>
                                <input
                                    type="text"
                                    value={searchValue}
                                    onChange={handleSearchChange}
                                    placeholder="Search in messages, sources, IP, context..."
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                />
                            </div>

                            {/* Log Type */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Log Type
                                </label>
                                <select
                                    value={localFilters.type}
                                    onChange={(e) => updateFilters({ type: e.target.value })}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                >
                                    <option value="all">All Types</option>
                                    <option value="error">Error</option>
                                    <option value="warning">Warning</option>
                                    <option value="info">Info</option>
                                </select>
                            </div>

                            {/* Source */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Source
                                </label>
                                <select
                                    value={localFilters.source}
                                    onChange={(e) => updateFilters({ source: e.target.value })}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                >
                                    <option value="all">All Sources</option>
                                    {sources.map((source) => (
                                        <option key={source} value={source}>{source}</option>
                                    ))}
                                </select>
                            </div>

                            {/* User */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    User
                                </label>
                                <select
                                    value={localFilters.user_id}
                                    onChange={(e) => updateFilters({ user_id: e.target.value })}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                >
                                    <option value="all">All Users</option>
                                    {users.map((user) => (
                                        <option key={user.id} value={user.id}>{user.name}</option>
                                    ))}
                                </select>
                            </div>

                            {/* Date Range */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Date From
                                </label>
                                <input
                                    type="date"
                                    value={localFilters.date_from}
                                    onChange={(e) => updateFilters({ date_from: e.target.value })}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                />
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Date To
                                </label>
                                <input
                                    type="date"
                                    value={localFilters.date_to}
                                    onChange={(e) => updateFilters({ date_to: e.target.value })}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                />
                            </div>

                            {/* Per Page */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Per Page
                                </label>
                                <select
                                    value={localFilters.per_page}
                                    onChange={(e) => updateFilters({ per_page: e.target.value })}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                >
                                    <option value="25">25</option>
                                    <option value="50">50</option>
                                    <option value="100">100</option>
                                    <option value="200">200</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    {/* Logs Table */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                        <div className="overflow-x-auto">
                            <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                <thead className="bg-gray-50 dark:bg-gray-700">
                                    <tr>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Type
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Source
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Message
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            User
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            IP Address
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Date
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                            Actions
                                        </th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {logs.data.map((log) => (
                                        <tr key={log.id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <span className={`px-2 py-1 rounded-full text-xs font-medium ${getLogTypeColor(log.type)}`}>
                                                    {log.type.toUpperCase()}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                {log.source}
                                            </td>
                                            <td className="px-6 py-4 text-sm text-gray-900 dark:text-white">
                                                <div className="max-w-xs">
                                                    <div className="mb-1">
                                                        {truncateMessage(log.message)}
                                                    </div>
                                                    {log.context_count > 0 && (
                                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                                            <span className="font-medium">Context:</span> {log.context_count} items
                                                            <button
                                                                onClick={() => {
                                                                    navigator.clipboard.writeText(log.formatted_context);
                                                                    toast.success('Context JSON copied!');
                                                                }}
                                                                className="ml-2 text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                                                title="Copy context JSON"
                                                            >
                                                                <i className="fas fa-copy"></i>
                                                            </button>
                                                        </div>
                                                    )}
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                {log.user_name || 'System'}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                                {log.ip_address || '-'}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600 dark:text-gray-400">
                                                {log.formatted_created_at}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <Link
                                                    href={route('admin.logs.show', log.id)}
                                                    className="text-blue-600 dark:text-blue-400 hover:text-blue-900 dark:hover:text-blue-300"
                                                >
                                                    View
                                                </Link>
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {logs.links && logs.links.length > 3 && (
                            <div className="bg-white dark:bg-gray-800 px-4 py-3 border-t border-gray-200 dark:border-gray-700 sm:px-6">
                                <div className="flex items-center justify-between">
                                    <div className="flex-1 flex justify-between sm:hidden">
                                        {logs.prev_page_url && (
                                            <Link
                                                href={logs.prev_page_url}
                                                className="relative inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600"
                                            >
                                                Previous
                                            </Link>
                                        )}
                                        {logs.next_page_url && (
                                            <Link
                                                href={logs.next_page_url}
                                                className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600"
                                            >
                                                Next
                                            </Link>
                                        )}
                                    </div>
                                    <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                                        <div>
                                            <p className="text-sm text-gray-700 dark:text-gray-300">
                                                Showing <span className="font-medium">{logs.from}</span> to{' '}
                                                <span className="font-medium">{logs.to}</span> of{' '}
                                                <span className="font-medium">{logs.total}</span> results
                                            </p>
                                        </div>
                                        <div>
                                            <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">
                                                {logs.links.map((link, index) => (
                                                    <Link
                                                        key={index}
                                                        href={link.url}
                                                        className={`relative inline-flex items-center px-4 py-2 border text-sm font-medium ${
                                                            link.active
                                                                ? 'z-10 bg-blue-50 dark:bg-blue-900/30 border-blue-500 text-blue-600 dark:text-blue-400'
                                                                : 'bg-white dark:bg-gray-700 border-gray-300 dark:border-gray-600 text-gray-500 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-600'
                                                        }`}
                                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                                    />
                                                ))}
                                            </nav>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        )}
                    </div>
                </div>

                {/* Clear Logs Modal */}
                {showClearModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white dark:bg-gray-800">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">Clear Old Logs</h3>
                                <div className="space-y-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Delete logs older than (days)
                                        </label>
                                        <input
                                            type="number"
                                            min="1"
                                            max="365"
                                            value={clearForm.days}
                                            onChange={(e) => setClearForm({ ...clearForm, days: parseInt(e.target.value) })}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Log Type (optional)
                                        </label>
                                        <select
                                            value={clearForm.type}
                                            onChange={(e) => setClearForm({ ...clearForm, type: e.target.value })}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        >
                                            <option value="all">All Types</option>
                                            <option value="error">Error Only</option>
                                            <option value="warning">Warning Only</option>
                                            <option value="info">Info Only</option>
                                        </select>
                                    </div>
                                </div>
                                <div className="flex justify-end space-x-3 mt-6">
                                    <button
                                        onClick={() => setShowClearModal(false)}
                                        className="px-4 py-2 bg-gray-300 dark:bg-gray-600 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-400 dark:hover:bg-gray-500 transition-colors"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleClearLogs}
                                        className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
                                    >
                                        Clear Logs
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}

                {/* Export Modal */}
                {showExportModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white dark:bg-gray-800">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">Export Logs</h3>
                                <div className="space-y-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Format
                                        </label>
                                        <select
                                            value={exportForm.format}
                                            onChange={(e) => setExportForm({ ...exportForm, format: e.target.value })}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        >
                                            <option value="csv">CSV</option>
                                            <option value="json">JSON</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Log Type (optional)
                                        </label>
                                        <select
                                            value={exportForm.type}
                                            onChange={(e) => setExportForm({ ...exportForm, type: e.target.value })}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        >
                                            <option value="all">All Types</option>
                                            <option value="error">Error Only</option>
                                            <option value="warning">Warning Only</option>
                                            <option value="info">Info Only</option>
                                        </select>
                                    </div>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Date From
                                            </label>
                                            <input
                                                type="date"
                                                value={exportForm.date_from}
                                                onChange={(e) => setExportForm({ ...exportForm, date_from: e.target.value })}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                            />
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Date To
                                            </label>
                                            <input
                                                type="date"
                                                value={exportForm.date_to}
                                                onChange={(e) => setExportForm({ ...exportForm, date_to: e.target.value })}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                            />
                                        </div>
                                    </div>
                                </div>
                                <div className="flex justify-end space-x-3 mt-6">
                                    <button
                                        onClick={() => setShowExportModal(false)}
                                        className="px-4 py-2 bg-gray-300 dark:bg-gray-600 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-400 dark:hover:bg-gray-500 transition-colors"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleExportLogs}
                                        className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                                    >
                                        Export
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        </AdminLayout>
    );
};

export default LogsIndex; 