import AdminLayout from '@/Layouts/AdminLayout';
import { Head } from '@inertiajs/react';
import { useState, useEffect, useCallback } from 'react';
import axios from 'axios';

const CustomerTransactionStats = () => {
    const [stats, setStats] = useState(null);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const [filters, setFilters] = useState({
        period: 'daily',
        start_date: '',
        end_date: '',
        idn_no: '',
        payment_status: '',
        transaction_type: '',
        payment_method: '',
        limit: 50,
        sort_by: 'total_processed',
        sort_order: 'desc'
    });
    const [selectedCustomer, setSelectedCustomer] = useState(null);
    const [customerHistory, setCustomerHistory] = useState(null);
    const [historyLoading, setHistoryLoading] = useState(false);
    const [showHistoryModal, setShowHistoryModal] = useState(false);
    const [debounceTimer, setDebounceTimer] = useState(null);
    const [historyPage, setHistoryPage] = useState(1);
    const [historyPerPage, setHistoryPerPage] = useState(25);
    const [currentPage, setCurrentPage] = useState(1);
    const [itemsPerPage] = useState(25);

    const fetchStats = useCallback(async () => {
        // Validate custom date range
        if (filters.period === 'custom') {
            if (!filters.start_date || !filters.end_date) {
                setError('Please select both start and end dates for custom range.');
                return;
            }
            
            const startDate = new Date(filters.start_date);
            const endDate = new Date(filters.end_date);
            const today = new Date();
            today.setHours(23, 59, 59, 999); // End of today
            
            if (startDate > today) {
                setError('Start date cannot be in the future.');
                return;
            }
            
            if (endDate > today) {
                setError('End date cannot be in the future.');
                return;
            }
            
            if (startDate > endDate) {
                setError('Start date cannot be after end date.');
                return;
            }
        }

        // Validate specific date
        if (filters.period === 'specific_date') {
            if (!filters.start_date) {
                setError('Please select a date for specific date analysis.');
                return;
            }
            
            const selectedDate = new Date(filters.start_date);
            const today = new Date();
            today.setHours(23, 59, 59, 999); // End of today
            
            if (selectedDate > today) {
                setError('Selected date cannot be in the future.');
                return;
            }
        }

        setLoading(true);
        setError(null);
        try {
            const params = { ...filters };
            // Remove empty filters
            Object.keys(params).forEach(key => {
                if (params[key] === '' || params[key] === null) {
                    delete params[key];
                }
            });

            const response = await axios.get('/admin/customer-transaction-stats/api', { params });
            console.log('Customer stats response:', response.data);
            setStats(response.data);
        } catch (error) {
            console.error('Error fetching customer stats:', error);
            setError('Failed to load customer transaction statistics. Please try again.');
        } finally {
            setLoading(false);
        }
    }, [filters]);

    const debouncedFetchStats = useCallback(() => {
        // Clear existing timer
        if (debounceTimer) {
            clearTimeout(debounceTimer);
        }

        // Set new timer
        const timer = setTimeout(() => {
            fetchStats();
        }, 500); // 500ms delay

        setDebounceTimer(timer);
    }, [filters, debounceTimer, fetchStats]);

    // Client-side pagination function
    const getPaginatedTransactions = (transactions, page, perPage) => {
        if (!transactions) return { items: [], pagination: null };
        
        const startIndex = (page - 1) * perPage;
        const endIndex = startIndex + perPage;
        const items = transactions.slice(startIndex, endIndex);
        
        const totalPages = Math.ceil(transactions.length / perPage);
        
        return {
            items,
            pagination: {
                current_page: page,
                last_page: totalPages,
                per_page: perPage,
                total: transactions.length,
                from: startIndex + 1,
                to: Math.min(endIndex, transactions.length)
            }
        };
    };

    const fetchCustomerHistory = async (customerId) => {
        setHistoryLoading(true);
        try {
            // Only pass relevant filters for customer history API
            const params = {
                start_date: filters.start_date || null,
                end_date: filters.end_date || null,
                payment_status: filters.payment_status || null,
                transaction_type: filters.transaction_type || null,
                payment_method: filters.payment_method || null,
                sort_by: 'created_at',
                sort_order: 'desc'
            };
            
            // Remove empty filters
            Object.keys(params).forEach(key => {
                if (params[key] === '' || params[key] === null) {
                    delete params[key];
                }
            });

            const response = await axios.get(`/admin/customer-transaction-history/${customerId}`, { params });
            setCustomerHistory(response.data);
            setShowHistoryModal(true);
            setCurrentPage(1); // Reset to first page when opening modal
        } catch (error) {
            console.error('Error fetching customer history:', error);
            setError('Failed to load customer transaction history. Please try again.');
        } finally {
            setHistoryLoading(false);
        }
    };

    const handlePageChange = (page) => {
        setCurrentPage(page);
    };

    const renderPagination = (pagination) => {
        if (!pagination || pagination.last_page <= 1) return null;

        const pages = [];
        const maxVisiblePages = 5;
        let startPage = Math.max(1, pagination.current_page - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(pagination.last_page, startPage + maxVisiblePages - 1);

        if (endPage - startPage + 1 < maxVisiblePages) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        for (let i = startPage; i <= endPage; i++) {
            pages.push(i);
        }

        return (
            <div className="flex items-center justify-center space-x-1 mt-4">
                <button
                    className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 disabled:opacity-50"
                    disabled={pagination.current_page === 1}
                    onClick={() => handlePageChange(pagination.current_page - 1)}
                >
                    Previous
                </button>
                
                {startPage > 1 && (
                    <>
                        <button
                            className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200"
                            onClick={() => handlePageChange(1)}
                        >
                            1
                        </button>
                        {startPage > 2 && <span className="px-2">...</span>}
                    </>
                )}
                
                {pages.map(page => (
                    <button
                        key={page}
                        className={`px-3 py-1 rounded ${
                            page === pagination.current_page 
                                ? 'bg-indigo-600 text-white' 
                                : 'bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200'
                        }`}
                        onClick={() => handlePageChange(page)}
                    >
                        {page}
                    </button>
                ))}
                
                {endPage < pagination.last_page && (
                    <>
                        {endPage < pagination.last_page - 1 && <span className="px-2">...</span>}
                        <button
                            className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200"
                            onClick={() => handlePageChange(pagination.last_page)}
                        >
                            {pagination.last_page}
                        </button>
                    </>
                )}
                
                <button
                    className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 disabled:opacity-50"
                    disabled={pagination.current_page === pagination.last_page}
                    onClick={() => handlePageChange(pagination.current_page + 1)}
                >
                    Next
                </button>
            </div>
        );
    };

    useEffect(() => {
        // Initial load
        fetchStats();
    }, []); // Only run on mount

    useEffect(() => {
        // Only trigger API call for non-date filter changes
        const { start_date, end_date, ...otherFilters } = filters;
        
        // Debounced fetch for filter changes (excluding date inputs)
        debouncedFetchStats();
        
        // Cleanup timer on unmount
        return () => {
            if (debounceTimer) {
                clearTimeout(debounceTimer);
            }
        };
    }, [filters.period, filters.idn_no, filters.payment_status, filters.transaction_type, filters.payment_method, filters.sort_by, filters.sort_order, filters.limit]);

    // Separate effect for date changes
    useEffect(() => {
        // Only trigger if we have valid dates for custom range or specific date
        if (filters.period === 'custom' && filters.start_date && filters.end_date) {
            fetchStats();
        } else if (filters.period === 'specific_date' && filters.start_date) {
            fetchStats();
        }
    }, [filters.start_date, filters.end_date]);

    const handleFilterChange = (key, value) => {
        setFilters(prev => ({
            ...prev,
            [key]: value
        }));
    };

    const handleCustomerClick = (customer) => {
        setSelectedCustomer(customer);
        fetchCustomerHistory(customer.user_id);
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
        }).format(amount);
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'paid':
            case 'completed':
                return 'text-green-600 dark:text-green-400';
            case 'pending':
            case 'unpaid':
                return 'text-yellow-600 dark:text-yellow-400';
            case 'failed':
            case 'expired':
                return 'text-red-600 dark:text-red-400';
            case 'cancelled':
            case 'closed':
                return 'text-gray-600 dark:text-gray-400';
            default:
                return 'text-gray-600 dark:text-gray-400';
        }
    };

    const getStatusBadgeColor = (status) => {
        switch (status) {
            case 'paid':
            case 'completed':
                return 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400';
            case 'pending':
            case 'unpaid':
                return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400';
            case 'failed':
            case 'expired':
                return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            case 'cancelled':
            case 'closed':
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400';
            default:
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400';
        }
    };

    if (loading) {
        return (
            <AdminLayout>
                <Head title="Customer Transaction Statistics" />
                <div className="py-12">
                    <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                            <div className="p-6 text-gray-900 dark:text-gray-100">
                                <div className="flex justify-between items-center mb-6">
                                    <h2 className="text-2xl font-bold">Customer Transaction Statistics</h2>
                                </div>
                                <div className="flex items-center justify-center py-12">
                                    <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900 dark:border-white"></div>
                                    <span className="ml-3 text-gray-600 dark:text-gray-400">Loading statistics...</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </AdminLayout>
        );
    }

    return (
        <AdminLayout>
            <Head title="Customer Transaction Statistics" />
            
            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900 dark:text-gray-100">
                            <div className="flex justify-between items-center mb-6">
                                <h2 className="text-2xl font-bold">Customer Transaction Statistics</h2>
                                <div className="text-sm text-gray-600 dark:text-gray-400">
                                    {stats?.period ? `${stats.period.charAt(0).toUpperCase() + stats.period.slice(1)} Period` : 'All time'}
                                </div>
                            </div>

                            {error && (
                                <div className="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                                    {error}
                                </div>
                            )}

                            {/* Filters */}
                            <div className="mb-6 bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                <h3 className="text-lg font-semibold mb-4">Filters</h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Period
                                        </label>
                                        <select
                                            value={filters.period}
                                            onChange={(e) => handleFilterChange('period', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        >
                                            <option value="daily">Daily</option>
                                            <option value="weekly">Weekly</option>
                                            <option value="monthly">Monthly</option>
                                            <option value="specific_date">Specific Date</option>
                                            <option value="custom">Custom Range</option>
                                        </select>
                                    </div>
                                    {filters.period === 'specific_date' && (
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                Select Date
                                            </label>
                                            <input
                                                type="date"
                                                value={filters.start_date}
                                                onChange={(e) => handleFilterChange('start_date', e.target.value)}
                                                max={new Date().toISOString().split('T')[0]}
                                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                            />
                                        </div>
                                    )}
                                    {filters.period === 'custom' && (
                                        <>
                                            <div>
                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                    Start Date
                                                </label>
                                                <input
                                                    type="date"
                                                    value={filters.start_date}
                                                    onChange={(e) => handleFilterChange('start_date', e.target.value)}
                                                    max={new Date().toISOString().split('T')[0]}
                                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                                />
                                            </div>
                                            <div>
                                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                    End Date
                                                </label>
                                                <input
                                                    type="date"
                                                    value={filters.end_date}
                                                    onChange={(e) => handleFilterChange('end_date', e.target.value)}
                                                    min={filters.start_date || undefined}
                                                    max={new Date().toISOString().split('T')[0]}
                                                    disabled={!filters.start_date}
                                                    className={`w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white ${
                                                        !filters.start_date ? 'bg-gray-100 dark:bg-gray-700 cursor-not-allowed' : ''
                                                    }`}
                                                />
                                            </div>
                                        </>
                                    )}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            IDN No
                                        </label>
                                        <input
                                            type="text"
                                            value={filters.idn_no}
                                            onChange={(e) => handleFilterChange('idn_no', e.target.value)}
                                            placeholder="Enter IDN No"
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Payment Status
                                        </label>
                                        <select
                                            value={filters.payment_status}
                                            onChange={(e) => handleFilterChange('payment_status', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        >
                                            <option value="">All Statuses</option>
                                            <option value="paid">Paid</option>
                                            <option value="completed">Completed</option>
                                            <option value="pending">Pending</option>
                                            <option value="unpaid">Unpaid</option>
                                            <option value="failed">Failed</option>
                                            <option value="expired">Expired</option>
                                            <option value="cancelled">Cancelled</option>
                                            <option value="closed">Closed</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Transaction Type
                                        </label>
                                        <select
                                            value={filters.transaction_type}
                                            onChange={(e) => handleFilterChange('transaction_type', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        >
                                            <option value="">All Types</option>
                                            <option value="deposit">Deposit</option>
                                            <option value="withdrawal">Withdrawal</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Payment Method
                                        </label>
                                        <input
                                            type="text"
                                            value={filters.payment_method}
                                            onChange={(e) => handleFilterChange('payment_method', e.target.value)}
                                            placeholder="Enter payment method"
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Sort By
                                        </label>
                                        <select
                                            value={filters.sort_by}
                                            onChange={(e) => handleFilterChange('sort_by', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        >
                                            <option value="total_processed">Total Processed</option>
                                            <option value="transaction_count">Transaction Count</option>
                                            <option value="last_transaction">Last Transaction</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Sort Order
                                        </label>
                                        <select
                                            value={filters.sort_order}
                                            onChange={(e) => handleFilterChange('sort_order', e.target.value)}
                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-800 dark:text-white"
                                        >
                                            <option value="desc">Descending</option>
                                            <option value="asc">Ascending</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            {/* Summary Cards */}
                            {stats?.totals && (
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                        <div className="flex items-center justify-between mb-4">
                                            <h3 className="font-semibold text-gray-900 dark:text-white">Total Customers</h3>
                                            <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                                                <i className="fas fa-users text-blue-600 dark:text-blue-400"></i>
                                            </div>
                                        </div>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {stats.totals.total_customers}
                                        </p>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                            Active customers
                                        </p>
                                    </div>

                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                        <div className="flex items-center justify-between mb-4">
                                            <h3 className="font-semibold text-gray-900 dark:text-white">Total Processed</h3>
                                            <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-full">
                                                <i className="fas fa-dollar-sign text-green-600 dark:text-green-400"></i>
                                            </div>
                                        </div>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {stats.totals.formatted_total_processed}
                                        </p>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                            {stats.totals.total_transactions} transactions
                                        </p>
                                    </div>

                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                        <div className="flex items-center justify-between mb-4">
                                            <h3 className="font-semibold text-gray-900 dark:text-white">Net Revenue</h3>
                                            <div className="p-2 bg-emerald-100 dark:bg-emerald-900/30 rounded-full">
                                                <i className="fas fa-chart-line text-emerald-600 dark:text-emerald-400"></i>
                                            </div>
                                        </div>
                                        <p className={`text-2xl font-bold ${stats.totals.net_revenue >= 0 ? 'text-emerald-600 dark:text-emerald-400' : 'text-red-600 dark:text-red-400'}`}>
                                            {stats.totals.formatted_net_revenue}
                                        </p>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                            Deposits - Withdrawals
                                        </p>
                                    </div>

                                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                        <div className="flex items-center justify-between mb-4">
                                            <h3 className="font-semibold text-gray-900 dark:text-white">Success Rate</h3>
                                            <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                                                <i className="fas fa-chart-line text-blue-600 dark:text-blue-400"></i>
                                            </div>
                                        </div>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {stats.totals.success_rate}%
                                        </p>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                            {stats.totals.pending_transactions} pending, {stats.totals.failed_transactions} failed
                                        </p>
                                    </div>
                                </div>
                            )}

                            {/* Customer List */}
                            {stats?.customers && stats.customers.length > 0 ? (
                                <div className="bg-white dark:bg-gray-800 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                    <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                            Customer Statistics ({stats.customers.length} customers)
                                        </h3>
                                    </div>
                                    <div className="overflow-x-auto">
                                        <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                            <thead className="bg-gray-50 dark:bg-gray-700">
                                                <tr>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Customer
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Total Processed
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Deposits
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Withdrawals
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Net Revenue
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Transactions
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Success Rate
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Last Transaction
                                                    </th>
                                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                                        Actions
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                                {stats.customers.map((customer) => (
                                                    <tr key={customer.user_id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div>
                                                                <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                                    {customer.username}
                                                                </div>
                                                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                                                    ID: {customer.user_id} | IDN: {customer.idn_no || 'N/A'}
                                                                </div>
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                                {customer.formatted_total_processed}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-green-600 dark:text-green-400">
                                                                {customer.formatted_total_deposits}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-red-600 dark:text-red-400">
                                                                {customer.formatted_total_withdrawals}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className={`text-sm font-medium ${customer.net_revenue >= 0 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}`}>
                                                                {customer.formatted_net_revenue}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-gray-900 dark:text-white">
                                                                {customer.transaction_count}
                                                            </div>
                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                Avg: {customer.formatted_avg_transaction_amount}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                                {customer.success_rate}%
                                                            </div>
                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                {customer.successful_transactions}/{customer.transaction_count}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap">
                                                            <div className="text-sm text-gray-900 dark:text-white">
                                                                {new Date(customer.last_transaction).toLocaleDateString()}
                                                            </div>
                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                {new Date(customer.last_transaction).toLocaleTimeString()}
                                                            </div>
                                                        </td>
                                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                            <button
                                                                onClick={() => handleCustomerClick(customer)}
                                                                className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300"
                                                            >
                                                                View History
                                                            </button>
                                                        </td>
                                                    </tr>
                                                ))}
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            ) : stats && (
                                <div className="bg-white dark:bg-gray-800 rounded-lg shadow border border-gray-200 dark:border-gray-700 p-6">
                                    <div className="text-center">
                                        <i className="fas fa-users text-gray-400 text-4xl mb-4"></i>
                                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">No customers found</h3>
                                        <p className="text-gray-600 dark:text-gray-400">
                                            No customers match the current filters. Try adjusting your search criteria.
                                        </p>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            {/* Customer History Modal */}
            {showHistoryModal && customerHistory && (
                <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                    <div className="relative top-20 mx-auto p-5 border w-11/12 max-w-6xl shadow-lg rounded-md bg-white dark:bg-gray-800">
                        <div className="mt-3">
                            <div className="flex justify-between items-center mb-4">
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                    Transaction History - {customerHistory.customer.name} ({customerHistory.customer.idn_no})
                                </h3>
                                <button
                                    onClick={() => setShowHistoryModal(false)}
                                    className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                                >
                                    <i className="fas fa-times text-xl"></i>
                                </button>
                            </div>

                            {historyLoading ? (
                                <div className="flex items-center justify-center py-8">
                                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900 dark:border-white"></div>
                                </div>
                            ) : (
                                <div className="space-y-6">
                                    {/* Customer Info */}
                                    <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                                            <div>
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">Customer ID</div>
                                                <div className="text-lg font-semibold text-gray-900 dark:text-white">{customerHistory.customer.id}</div>
                                            </div>
                                            <div>
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">IDN No</div>
                                                <div className="text-lg font-semibold text-gray-900 dark:text-white">{customerHistory.customer.idn_no}</div>
                                            </div>
                                            <div>
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">Role</div>
                                                <div className="text-lg font-semibold text-gray-900 dark:text-white capitalize">{customerHistory.customer.role}</div>
                                            </div>
                                            <div>
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">Date Range</div>
                                                <div className="text-lg font-semibold text-gray-900 dark:text-white">{customerHistory.summary.date_range}</div>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Summary Cards */}
                                    <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                            <div className="flex items-center justify-between mb-2">
                                                <h4 className="font-semibold text-gray-900 dark:text-white">Total Processed</h4>
                                                <div className="p-2 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                                                    <i className="fas fa-exchange-alt text-blue-600 dark:text-blue-400 text-sm"></i>
                                                </div>
                                            </div>
                                            <p className="text-2xl font-bold text-gray-900 dark:text-white">${customerHistory.summary.formatted_total_amount}</p>
                                            <p className="text-sm text-gray-600 dark:text-gray-400">No. of Transactions: {customerHistory.summary.total_transactions}</p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400">Success Rate: {customerHistory.summary.overall_success_rate}%</p>
                                        </div>

                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                            <div className="flex items-center justify-between mb-2">
                                                <h4 className="font-semibold text-gray-900 dark:text-white">Deposits</h4>
                                                <div className="p-2 bg-green-100 dark:bg-green-900/30 rounded-full">
                                                    <i className="fas fa-arrow-up text-green-600 dark:text-green-400 text-sm"></i>
                                                </div>
                                            </div>
                                            <p className="text-2xl font-bold text-green-600 dark:text-green-400">{formatCurrency(customerHistory.deposits.formatted_successful_amount)}</p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400">Success: {customerHistory.deposits.successful_count}/{customerHistory.deposits.total_count} ({customerHistory.deposits.success_rate}%)</p>
                                        </div>

                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                            <div className="flex items-center justify-between mb-2">
                                                <h4 className="font-semibold text-gray-900 dark:text-white">Withdrawals</h4>
                                                <div className="p-2 bg-red-100 dark:bg-red-900/30 rounded-full">
                                                    <i className="fas fa-arrow-down text-red-600 dark:text-red-400 text-sm"></i>
                                                </div>
                                            </div>
                                            <p className="text-2xl font-bold text-red-600 dark:text-red-400">{formatCurrency(customerHistory.withdrawals.formatted_successful_amount)}</p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400">Success: {customerHistory.withdrawals.successful_count}/{customerHistory.withdrawals.total_count} ({customerHistory.withdrawals.success_rate}%)</p>
                                        </div>

                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                            <div className="flex items-center justify-between mb-2">
                                                <h4 className="font-semibold text-gray-900 dark:text-white">Net Revenue</h4>
                                                <div className="p-2 bg-emerald-100 dark:bg-emerald-900/30 rounded-full">
                                                    <i className="fas fa-chart-line text-emerald-600 dark:text-emerald-400 text-sm"></i>
                                                </div>
                                            </div>
                                            <p className={`text-2xl font-bold ${customerHistory.net_revenue.amount >= 0 ? 'text-emerald-600 dark:text-emerald-400' : 'text-red-600 dark:text-red-400'}`}>
                                                {formatCurrency(customerHistory.net_revenue.formatted_amount)}
                                            </p>
                                            <p className="text-sm text-gray-600 dark:text-gray-400">Successful transactions only</p>
                                        </div>
                                    </div>

                                    {/* Payment Methods Breakdown */}
                                    {customerHistory.payment_methods && customerHistory.payment_methods.length > 0 && (
                                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                            <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                                <h4 className="text-lg font-semibold text-gray-900 dark:text-white">Payment Methods Breakdown</h4>
                                            </div>
                                            <div className="overflow-x-auto">
                                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                                        <tr>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Method</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Total</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Successful</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Success Rate</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Deposits</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Withdrawals</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                                        {customerHistory.payment_methods.map((method) => (
                                                            <tr key={method.method} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                                                    {method.method}
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                    {method.total_transactions} ({formatCurrency(method.formatted_total_amount)})
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                    {method.successful_transactions} ({formatCurrency(method.formatted_successful_amount)})
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                                                                        method.success_rate >= 80 ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                                        method.success_rate >= 60 ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400' :
                                                                        'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                                                    }`}>
                                                                        {method.success_rate}%
                                                                    </span>
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400">
                                                                    {method.deposits_count} ({formatCurrency(method.formatted_deposits_amount)})
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-red-600 dark:text-red-400">
                                                                    {method.withdrawals_count} ({formatCurrency(method.formatted_withdrawals_amount)})
                                                                </td>
                                                            </tr>
                                                        ))}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    )}

                                    {/* Monthly Breakdown */}
                                    {customerHistory.monthly_breakdown && customerHistory.monthly_breakdown.length > 0 && (
                                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                            <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                                <h4 className="text-lg font-semibold text-gray-900 dark:text-white">Monthly Breakdown</h4>
                                            </div>
                                            <div className="overflow-x-auto">
                                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                                        <tr>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Month</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Total</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Successful</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Success Rate</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Deposits</th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Withdrawals</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                                        {customerHistory.monthly_breakdown.map((month) => (
                                                            <tr key={month.month} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                                                    {month.month_label}
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                    {month.total_transactions} ({formatCurrency(month.formatted_total_amount)})
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                    {month.successful_transactions} ({formatCurrency(month.formatted_successful_amount)})
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                                                                        month.success_rate >= 80 ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                                        month.success_rate >= 60 ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400' :
                                                                        'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                                                    }`}>
                                                                        {month.success_rate}%
                                                                    </span>
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-green-600 dark:text-green-400">
                                                                    {month.deposits_count} ({formatCurrency(month.formatted_deposits_amount)})
                                                                </td>
                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-red-600 dark:text-red-400">
                                                                    {month.withdrawals_count} ({formatCurrency(month.formatted_withdrawals_amount)})
                                                                </td>
                                                            </tr>
                                                        ))}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    )}

                                    {/* Transactions Table with Client-side Pagination */}
                                    {customerHistory && customerHistory.transactions && (
                                        <>
                                            {(() => {
                                                const paginatedData = getPaginatedTransactions(customerHistory.transactions, currentPage, itemsPerPage);
                                                return (
                                                    <>
                                                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow border border-gray-200 dark:border-gray-700">
                                                            <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                                                                <h4 className="text-lg font-semibold text-gray-900 dark:text-white">
                                                                    Transaction Details ({paginatedData.pagination?.total || 0} transactions)
                                                                </h4>
                                                            </div>
                                                            <div className="overflow-x-auto">
                                                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                                                        <tr>
                                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Invoice ID</th>
                                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Amount</th>
                                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Type</th>
                                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Payment Method</th>
                                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Status</th>
                                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Date</th>
                                                                        </tr>
                                                                    </thead>
                                                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                                                        {paginatedData.items.map((transaction) => (
                                                                            <tr key={transaction.id} className="hover:bg-gray-50 dark:hover:bg-gray-700">
                                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                                    {transaction.invoice_id}
                                                                                </td>
                                                                                <td className="px-4 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">
                                                                                    {transaction.formatted_amount}
                                                                                </td>
                                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                                                                                        transaction.type === 'deposit' 
                                                                                            ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400'
                                                                                            : 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                                                                    }`}>
                                                                                        {transaction.type}
                                                                                    </span>
                                                                                </td>
                                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                                    {transaction.payment_method}
                                                                                </td>
                                                                                <td className="px-4 py-4 whitespace-nowrap">
                                                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                                                                                        transaction.is_successful ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                                                        transaction.is_pending ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400' :
                                                                                        'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                                                                    }`}>
                                                                                        {transaction.payment_status}
                                                                                    </span>
                                                                                </td>
                                                                                <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                                                                    {new Date(transaction.created_at).toLocaleDateString()}
                                                                                    <br />
                                                                                    <span className="text-xs text-gray-500 dark:text-gray-400">
                                                                                        {new Date(transaction.created_at).toLocaleTimeString()}
                                                                                    </span>
                                                                                </td>
                                                                            </tr>
                                                                        ))}
                                                                    </tbody>
                                                                </table>
                                                            </div>
                                                        </div>
                                                        
                                                        {/* Pagination Controls */}
                                                        {renderPagination(paginatedData.pagination)}
                                                    </>
                                                );
                                            })()}
                                        </>
                                    )}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            )}
        </AdminLayout>
    );
};

export default CustomerTransactionStats; 