import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { useState, useEffect, useCallback } from 'react';
import axios from 'axios';

const CustomerTransactionHistory = ({ customer }) => {
    const [stats, setStats] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [filters, setFilters] = useState({
        start_date: '',
        end_date: '',
        payment_status: '',
        transaction_type: '',
        payment_method: '',
        sort_by: 'created_at',
        sort_order: 'desc'
    });
    const [currentPage, setCurrentPage] = useState(1);
    const [itemsPerPage] = useState(25);

    const fetchData = useCallback(async () => {
        setLoading(true);
        setError(null);
        try {
            const params = { ...filters };
            // Remove empty filters
            Object.keys(params).forEach(key => {
                if (params[key] === '' || params[key] === null) {
                    delete params[key];
                }
            });

            const response = await axios.get(`/admin/customer-transaction-history/${customer.id}`, { params });
            setStats(response.data);
        } catch (error) {
            console.error('Error fetching customer history:', error);
            setError('Failed to load customer transaction history. Please try again.');
        } finally {
            setLoading(false);
        }
    }, [filters, customer.id]);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const handleFilterChange = (key, value) => {
        setFilters(prev => ({ ...prev, [key]: value }));
        setCurrentPage(1); // Reset to first page when filters change
    };

    const clearFilters = () => {
        setFilters({
            start_date: '',
            end_date: '',
            payment_status: '',
            transaction_type: '',
            payment_method: '',
            sort_by: 'created_at',
            sort_order: 'desc'
        });
        setCurrentPage(1);
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'paid':
            case 'completed':
                return 'text-green-600 dark:text-green-400';
            case 'pending':
            case 'unpaid':
                return 'text-yellow-600 dark:text-yellow-400';
            case 'failed':
            case 'expired':
            case 'cancelled':
            case 'closed':
                return 'text-red-600 dark:text-red-400';
            default:
                return 'text-gray-600 dark:text-gray-400';
        }
    };

    const getStatusBadgeColor = (status) => {
        switch (status) {
            case 'paid':
            case 'completed':
                return 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400';
            case 'pending':
            case 'unpaid':
                return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400';
            case 'failed':
            case 'expired':
            case 'cancelled':
            case 'closed':
                return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            default:
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400';
        }
    };

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    // Client-side pagination
    const getPaginatedTransactions = (transactions, page, perPage) => {
        if (!transactions) return { items: [], pagination: null };
        
        const startIndex = (page - 1) * perPage;
        const endIndex = startIndex + perPage;
        const items = transactions.slice(startIndex, endIndex);
        
        const totalPages = Math.ceil(transactions.length / perPage);
        
        return {
            items,
            pagination: {
                current_page: page,
                last_page: totalPages,
                per_page: perPage,
                total: transactions.length,
                from: startIndex + 1,
                to: Math.min(endIndex, transactions.length)
            }
        };
    };

    const { items: paginatedTransactions, pagination } = getPaginatedTransactions(
        stats?.transactions || [], 
        currentPage, 
        itemsPerPage
    );

    const handlePageChange = (page) => {
        setCurrentPage(page);
    };

    const renderPagination = () => {
        if (!pagination || pagination.last_page <= 1) return null;

        const pages = [];
        const maxVisiblePages = 5;
        let startPage = Math.max(1, pagination.current_page - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(pagination.last_page, startPage + maxVisiblePages - 1);

        if (endPage - startPage + 1 < maxVisiblePages) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        for (let i = startPage; i <= endPage; i++) {
            pages.push(i);
        }

        return (
            <div className="flex items-center justify-center space-x-1 mt-4 mb-4">
                <button
                    className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 disabled:opacity-50"
                    disabled={pagination.current_page === 1}
                    onClick={() => handlePageChange(pagination.current_page - 1)}
                >
                    Previous
                </button>
                
                {startPage > 1 && (
                    <>
                        <button
                            className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200"
                            onClick={() => handlePageChange(1)}
                        >
                            1
                        </button>
                        {startPage > 2 && <span className="px-2">...</span>}
                    </>
                )}
                
                {pages.map(page => (
                    <button
                        key={page}
                        className={`px-3 py-1 rounded ${
                            page === pagination.current_page
                                ? 'bg-[#d34a40] text-white'
                                : 'bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200'
                        }`}
                        onClick={() => handlePageChange(page)}
                    >
                        {page}
                    </button>
                ))}
                
                {endPage < pagination.last_page && (
                    <>
                        {endPage < pagination.last_page - 1 && <span className="px-2">...</span>}
                        <button
                            className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200"
                            onClick={() => handlePageChange(pagination.last_page)}
                        >
                            {pagination.last_page}
                        </button>
                    </>
                )}
                
                <button
                    className="px-3 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 disabled:opacity-50"
                    disabled={pagination.current_page === pagination.last_page}
                    onClick={() => handlePageChange(pagination.current_page + 1)}
                >
                    Next
                </button>
            </div>
        );
    };

    if (loading) {
        return (
            <AdminLayout>
                <Head title={`Transaction History - ${customer.name}`} />
                <div className="py-12">
                    <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                        <div className="flex items-center justify-center">
                            <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-[#d34a40]"></div>
                        </div>
                    </div>
                </div>
            </AdminLayout>
        );
    }

    if (error) {
        return (
            <AdminLayout>
                <Head title={`Transaction History - ${customer.name}`} />
                <div className="py-12">
                    <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                        <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                            {error}
                        </div>
                    </div>
                </div>
            </AdminLayout>
        );
    }

    return (
        <AdminLayout>
            <Head title={`Transaction History - ${customer.name}`} />
            
            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header Section */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] p-6 rounded-[15px] shadow-lg mb-6">
                        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
                            <div>
                                <h1 className="text-3xl font-bold text-white">Transaction History</h1>
                                <p className="text-gray-100 dark:text-gray-200 mt-2">
                                    {customer.name} ({customer.idn_no})
                                </p>
                            </div>
                            <div className="flex gap-3">
                                <a
                                    href={route('admin.customer-transaction-history.download', customer.id)}
                                    className="bg-purple-600 text-white px-6 py-3 rounded-lg hover:bg-purple-700 transition-colors flex items-center shadow-md"
                                    target="_blank"
                                    rel="noopener noreferrer"
                                >
                                    <i className="fas fa-download mr-2"></i>
                                    Download Report
                                </a>
                                <Link
                                    href={route('admin.users.index')}
                                    className="bg-white text-[#d34a40] px-6 py-3 rounded-lg hover:bg-gray-100 transition-colors flex items-center shadow-md"
                                >
                                    <i className="fas fa-arrow-left mr-2"></i>
                                    Back to Users
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Customer Info */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6 mb-6">
                        <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-4">Customer Information</h2>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label className="block text-sm font-medium text-gray-500 dark:text-gray-400">Name</label>
                                <p className="text-lg font-semibold text-gray-900 dark:text-white">{customer.name}</p>
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-500 dark:text-gray-400">IDN No</label>
                                <p className="text-lg font-semibold text-gray-900 dark:text-white">{customer.idn_no}</p>
                            </div>
                        </div>
                    </div>

                    {/* Summary Statistics */}
                    {stats && (
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-6">
                            <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6">
                                <div className="flex items-center">
                                    <div className="p-3 rounded-full bg-blue-100 dark:bg-blue-900/30">
                                        <i className="fas fa-exchange-alt text-blue-600 dark:text-blue-400"></i>
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Total Transactions</p>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">{stats.summary.total_transactions}</p>
                                    </div>
                                </div>
                            </div>

                            <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6">
                                <div className="flex items-center">
                                    <div className="p-3 rounded-full bg-green-100 dark:bg-green-900/30">
                                        <i className="fas fa-arrow-up text-green-600 dark:text-green-400"></i>
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Total Deposits</p>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {formatCurrency(stats.deposits.total_amount)}
                                        </p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                            {stats.deposits.total_count} transactions
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6">
                                <div className="flex items-center">
                                    <div className="p-3 rounded-full bg-emerald-100 dark:bg-emerald-900/30">
                                        <i className="fas fa-check-circle text-emerald-600 dark:text-emerald-400"></i>
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Success Amount</p>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {formatCurrency(stats.deposits.successful_amount + stats.withdrawals.successful_amount)}
                                        </p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                            {stats.deposits.successful_count + stats.withdrawals.successful_count} transactions
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6">
                                <div className="flex items-center">
                                    <div className="p-3 rounded-full bg-yellow-100 dark:bg-yellow-900/30">
                                        <i className="fas fa-clock text-yellow-600 dark:text-yellow-400"></i>
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Pending Amount</p>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {formatCurrency(stats.deposits.pending_amount + stats.withdrawals.pending_amount)}
                                        </p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                            {stats.deposits.pending_count + stats.withdrawals.pending_count} transactions
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6">
                                <div className="flex items-center">
                                    <div className="p-3 rounded-full bg-red-100 dark:bg-red-900/30">
                                        <i className="fas fa-times-circle text-red-600 dark:text-red-400"></i>
                                    </div>
                                    <div className="ml-4">
                                        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">Failed Amount</p>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {formatCurrency(stats.deposits.failed_amount + stats.withdrawals.failed_amount)}
                                        </p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                            {stats.deposits.failed_count + stats.withdrawals.failed_count} transactions
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Detailed Breakdown */}
                    {stats && (
                        <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6 mb-6">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Transaction Status Breakdown</h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                {/* Success Transactions */}
                                <div className="border border-green-200 dark:border-green-800 rounded-lg p-4">
                                    <div className="flex items-center mb-3">
                                        <div className="p-2 rounded-full bg-green-100 dark:bg-green-900/30">
                                            <i className="fas fa-check-circle text-green-600 dark:text-green-400"></i>
                                        </div>
                                        <h4 className="ml-3 font-semibold text-green-800 dark:text-green-400">Success</h4>
                                    </div>
                                    <div className="space-y-2">
                                        <div className="flex justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400">Deposits:</span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">{stats.deposits.formatted_successful_amount}</span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400">Withdrawals:</span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">{stats.withdrawals.formatted_successful_amount}</span>
                                        </div>
                                        <div className="flex justify-between border-t pt-2">
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">Total:</span>
                                            <span className="text-sm font-bold text-green-600 dark:text-green-400">
                                                {formatCurrency(stats.deposits.successful_amount + stats.withdrawals.successful_amount)}
                                            </span>
                                        </div>
                                    </div>
                                </div>

                                {/* Pending Transactions */}
                                <div className="border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
                                    <div className="flex items-center mb-3">
                                        <div className="p-2 rounded-full bg-yellow-100 dark:bg-yellow-900/30">
                                            <i className="fas fa-clock text-yellow-600 dark:text-yellow-400"></i>
                                        </div>
                                        <h4 className="ml-3 font-semibold text-yellow-800 dark:text-yellow-400">Pending</h4>
                                    </div>
                                    <div className="space-y-2">
                                        <div className="flex justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400">Deposits:</span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">{stats.deposits.formatted_pending_amount}</span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400">Withdrawals:</span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">{stats.withdrawals.formatted_pending_amount}</span>
                                        </div>
                                        <div className="flex justify-between border-t pt-2">
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">Total:</span>
                                            <span className="text-sm font-bold text-yellow-600 dark:text-yellow-400">
                                                {formatCurrency(stats.deposits.pending_amount + stats.withdrawals.pending_amount)}
                                            </span>
                                        </div>
                                    </div>
                                </div>

                                {/* Failed/Expired/Cancelled Transactions */}
                                <div className="border border-red-200 dark:border-red-800 rounded-lg p-4">
                                    <div className="flex items-center mb-3">
                                        <div className="p-2 rounded-full bg-red-100 dark:bg-red-900/30">
                                            <i className="fas fa-times-circle text-red-600 dark:text-red-400"></i>
                                        </div>
                                        <h4 className="ml-3 font-semibold text-red-800 dark:text-red-400">Failed/Expired/Cancelled</h4>
                                    </div>
                                    <div className="space-y-2">
                                        <div className="flex justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400">Deposits:</span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">{stats.deposits.formatted_failed_amount}</span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-sm text-gray-600 dark:text-gray-400">Withdrawals:</span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">{stats.withdrawals.formatted_failed_amount}</span>
                                        </div>
                                        <div className="flex justify-between border-t pt-2">
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">Total:</span>
                                            <span className="text-sm font-bold text-red-600 dark:text-red-400">
                                                {formatCurrency(stats.deposits.failed_amount + stats.withdrawals.failed_amount)}
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Filters */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg p-6 mb-6">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Filters</h3>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Start Date</label>
                                <input
                                    type="date"
                                    value={filters.start_date}
                                    onChange={(e) => handleFilterChange('start_date', e.target.value)}
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                />
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">End Date</label>
                                <input
                                    type="date"
                                    value={filters.end_date}
                                    onChange={(e) => handleFilterChange('end_date', e.target.value)}
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                />
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Status</label>
                                <select
                                    value={filters.payment_status}
                                    onChange={(e) => handleFilterChange('payment_status', e.target.value)}
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                >
                                    <option value="">All Statuses</option>
                                    <option value="paid">Paid</option>
                                    <option value="completed">Completed</option>
                                    <option value="pending">Pending</option>
                                    <option value="unpaid">Unpaid</option>
                                    <option value="failed">Failed</option>
                                    <option value="expired">Expired</option>
                                    <option value="cancelled">Cancelled</option>
                                    <option value="closed">Closed</option>
                                </select>
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Type</label>
                                <select
                                    value={filters.transaction_type}
                                    onChange={(e) => handleFilterChange('transaction_type', e.target.value)}
                                    className="w-full rounded-lg border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-[#d34a40] focus:ring-[#d34a40]"
                                >
                                    <option value="">All Types</option>
                                    <option value="deposit">Deposit</option>
                                    <option value="withdrawal">Withdrawal</option>
                                </select>
                            </div>
                        </div>
                        <div className="flex justify-end mt-4">
                            <button
                                onClick={clearFilters}
                                className="px-4 py-2 bg-gray-100 text-gray-600 rounded-lg hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600"
                            >
                                Clear Filters
                            </button>
                        </div>
                    </div>

                    {/* Transactions Table */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg overflow-hidden">
                        <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Transactions ({pagination?.total || 0})
                            </h3>
                        </div>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead className="bg-gray-50 dark:bg-gray-700">
                                    <tr>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">ID</th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Amount</th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Type</th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Status</th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Method</th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Date</th>
                                    </tr>
                                </thead>
                                <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    {paginatedTransactions.map((transaction) => (
                                        <tr key={transaction.id} className="hover:bg-gray-50 dark:hover:bg-gray-700/50">
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-gray-100">
                                                {transaction.invoice_id || transaction.id}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-gray-100">
                                                {transaction.formatted_amount}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                                                    transaction.type === 'deposit' 
                                                        ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400'
                                                        : 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                                }`}>
                                                    {transaction.type.charAt(0).toUpperCase() + transaction.type.slice(1)}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadgeColor(transaction.payment_status)}`}>
                                                    {transaction.payment_status.charAt(0).toUpperCase() + transaction.payment_status.slice(1)}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-gray-100">
                                                {transaction.payment_method}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-gray-100">
                                                {new Date(transaction.created_at).toLocaleDateString()}
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                        {renderPagination()}
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default CustomerTransactionHistory; 