<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use App\Helpers\PayUtils;

class WalletService
{
    protected $apiService;

    public function __construct(ApiService $apiService)
    {
        $this->apiService = $apiService;
    }

    public function getMerchantBalance($gateway = 'tryspeed')
    {
        if ($gateway === 'ggpayout') {
            return $this->getGGPayoutMerchantBalance();
        } else {
            return $this->getTrySpeedMerchantBalance();
        }
    }
    
    protected function getTrySpeedMerchantBalance()
    {
        try {
            // Cache the balance for 5 minutes to avoid too many API calls
            return Cache::remember('tryspeed_merchant_balance', 300, function () {
                $response = $this->apiService->get('https://api.tryspeed.com/balances', [
                    'Authorization' => 'Basic ' . env('TRYSPEED_API_KEY'),
                    'accept' => 'application/json'
                ]);

                if (isset($response['status']) && $response['status'] === 'error') {
                    Log::error('Failed to fetch TrySpeed merchant balance', ['error' => $response['message']]);
                    return null;
                }

                // Format the response to match the expected structure
                if (isset($response['object']) && $response['object'] === 'balance' && isset($response['available']) && is_array($response['available'])) {
                    $formattedBalance = [];
                    foreach ($response['available'] as $balance) {
                        if (isset($balance['target_currency']) && isset($balance['amount'])) {
                            $formattedBalance[$balance['target_currency']] = (float) $balance['amount'];
                        }
                    }
                    // Only return if we have valid balance data
                    return !empty($formattedBalance) ? $formattedBalance : null;
                }

                Log::error('Unexpected TrySpeed balance response format', ['response' => $response]);
                return null;
            });
        } catch (\Exception $e) {
            Log::error('Error fetching TrySpeed merchant balance', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    protected function getGGPayoutMerchantBalance()
    {
        try {
            // Cache the balance for 5 seconds to match API limit
            return Cache::remember('ggpayout_merchant_balance', 5, function () {
                $url = 'https://api.ggpayout.com/api/balance/query';
                
                $data = [
                    'mchNo' => env('GGPAYOUT_MERCHANT_ID'),
                    'timestamp' => round(microtime(true) * 1000), // 13-digit timestamp
                    'signType' => 'MD5'
                ];
                
                // Sign the request
                $apiKey = env('GGPAYOUT_API_KEY');
                $signedData = PayUtils::sign($data, $apiKey);
                
                $response = $this->apiService->post($url, $signedData);
                
                if (!isset($response['code']) || $response['code'] !== 0) {
                    Log::error('Failed to fetch GGPayout merchant balance', ['response' => $response]);
                    return null;
                }

                // Format the response to match the expected structure
                if (isset($response['data']['accounts']) && is_array($response['data']['accounts'])) {
                    $formattedBalance = [];
                    foreach ($response['data']['accounts'] as $account) {
                        if (isset($account['currency']) && isset($account['availableBalance'])) {
                            $formattedBalance[strtoupper($account['currency'])] = [
                                'balance' => (float) $account['balance'] / 100,
                                'availableBalance' => (float) $account['availableBalance'] / 100,
                                'transferPendingAmount' => (float) $account['transferPendingAmount'] / 100,
                                'delayAmount' => (float) $account['delayAmount'] / 100
                            ];
                        }
                    }
                    return !empty($formattedBalance) ? $formattedBalance : null;
                }

                Log::error('Unexpected GGPayout balance response format', ['response' => $response]);
                return null;
            });
        } catch (\Exception $e) {
            Log::error('Error fetching GGPayout merchant balance', ['error' => $e->getMessage()]);
            return null;
        }
    }
}