<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Transaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_id',
        'user_id',
        'username',
        'game_id',
        'amount',
        'target_amount',
        'target_amount_paid',
        'exchange_rate',
        'currency',
        'target_currency',
        'type',
        'payment_gateway',
        'mchOrderNo',
        'payment_status',
        'transaction_id',
        'payment_method',
        'telegram_message_id',
        'telegram_chat_id',
        'telegram_poll_id',
        'payment_address',
        'processed_at',
        'processed_by'
    ];

    protected $casts = [
        'telegram_message_id' => 'string',
        'telegram_chat_id'    => 'string',
        'processed_at' => 'datetime',
    ];

    /**
     * Relationship: Transaction belongs to a User
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relationship: Transaction belongs to a Game (nullable)
     */
    public function game()
    {
        return $this->belongsTo(Game::class);
    }

    /**
     * Scope: Filter transactions by type
     */
    public function scopeType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope: Filter transactions by payment status
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    // Auto-generate invoice_id starting from 100001
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($transaction) {
            $lastInvoice = Transaction::max('invoice_id');
            $transaction->invoice_id = $lastInvoice ? $lastInvoice + 1 : 100001;
        });
    }

    public function markAsProcessed($status, $processedBy)
    {
        $this->update([
            'payment_status' => $status,
            'processed_at' => now(),
            'processed_by' => $processedBy
        ]);
    }
}
