<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Wallet;
use App\Models\Game;
use Illuminate\Support\Facades\DB;
use Telegram\Bot\Laravel\Facades\Telegram;
use App\Services\PaymentService;

class WebhookController extends Controller
{
    protected $chatId;
    protected $paymentService;
    public function __construct(PaymentService $paymentService)
    {
        $this->chatId = env('TELEGRAM_CHAT_ID_PAID'); // Define chat ID from .env
        $this->paymentService = $paymentService;
    }

    public function handlePaymentWebhook(Request $request)
    {

        $eventType = $request->input('event_type');
        $data = $request->input('data.object');
        if($data['metadata']['payment_from'] == 'vegasfish'){
            logEvent('info', 'WebhookController', 'Webhook received: Event "' . $eventType . '" for Payment ID ' . $data['id'], $request->all());

            $eventHandlers = [
                'payment.created'   => 'handlePaymentCreated',
                'payment.paid'      => 'handlePaymentPaid',
                'payment.confirmed' => 'handlePaymentConfirmed',
                'payment.expired'   => 'handlePaymentExpired',
                'payment.cancelled' => 'handlePaymentCancelled',
                'withdraw.created'  => 'handleWithdrawCreated',
                'withdraw.paid'     => 'handleWithdrawPaid',
                'withdraw.failed'   => 'handleWithdrawFailed',
            ];

            if (isset($eventHandlers[$eventType])) {
                return $this->{$eventHandlers[$eventType]}($data);
            }

            logEvent('warning', 'WebhookController', 'Unhandled event type', ['event_type' => $eventType]);
            return response()->json(['message' => 'Event type not handled'], 200);
        }
    }

    private function handlePaymentCreated($data)
    {
        $userId = $data['metadata']['user_id'] ?? null;
        $gameId = $data['metadata']['game_id'] ?? null;
        $username = $data['metadata']['username'] ?? null;
        //$amount = $data['exchange_rate'] * $data['target_amount'];
        $target_amount = $data['target_amount'];
        $exchange_rate = $data['exchange_rate'];
        $target_amount_paid = $data['target_amount_paid'];
        $amount = $data['amount'];
        
        logEvent('info', 'WebhookController', 'Processing payment.created event for payment Id ' . $data['id'], ['payment_id' => $data['id']], $userId);
        Transaction::updateOrCreate(
            [
                'transaction_id' => $data['id'],
            ],
            [
            'user_id' => $userId,
            'username' => $username,
            'game_id' => $gameId,
            'amount' => $amount,
            'target_amount' => $target_amount,
            'exchange_rate' => $exchange_rate,
            'target_amount_paid' => $target_amount_paid,
            'currency' => $data['currency'],
            'type' => 'deposit',
            'payment_status' => $data['status'],
            'payment_method' => null,
        ]);

        return response()->json(['message' => 'Payment Created Processed'], 200);
    }

    private function handlePaymentPaid($data)
    {
        $userId = $data['metadata']['user_id'] ?? null;
        $IDNNo = $data['metadata']['idn_no'] ?? null;
        $gameId = $data['metadata']['game_id'] ?? null;
        $username = $data['metadata']['username'] ?? null;
        $game = Game::find($gameId);
        
        logEvent('info', 'WebhookController', 'Processing payment.paid event for payment Id ' . $data['id'], ['payment_id' => $data['id']], $userId);

        $transaction = Transaction::where('transaction_id', $data['id'])->first();
        //$withdraw_request = $transaction['payment_address'];
        //$response = $this->paymentService->WithdrawalDecode($data['target_amount_paid'], $withdraw_request);
        $amountFormatted = number_format((float)($data['target_amount_paid'] / $data['exchange_rate']), 2, '.', '');

        //if(abs($data['target_amount'] - $data['target_amount_paid']) <= 1){
            /*if ($transaction) {
                $amountFormatted = number_format((float)($data['target_amount_paid'] / $data['exchange_rate']), 2, '.', '');
                $transaction->update([
                    'payment_status' => $data['status'],
                    'amount' => $amountFormatted,
                    'payment_method' => $data['payment_method_paid_by'],
                    'username' => $username ?? $transaction->username,
                ]);
                //$this->updateWalletBalance($transaction->user_id, $transaction->amount);
            }*/
			
			if ($transaction) {
                $transaction->update([
                    'payment_status' => $data['status'],
                    'amount' => $amountFormatted,
                    'payment_method' => $data['payment_method_paid_by'],
                    'username' => $username ?? $transaction->username,
                ]);
            } else {
                $transaction = Transaction::where('mchOrderNo', $data['payment_request_paid_by'])->first();
                $transaction->update([
                    'transaction_id' => $data['id'],
                    'payment_status' => $data['status'],
                    'amount' => $amountFormatted,
                    'payment_method' => $data['payment_method_paid_by'],
                    'username' => $username ?? $transaction->username,
                    'mchOrderNo' => null
                ]);
            }

            $question = "Player ID: {$IDNNo}\n"
                . "Player Nickname: {$transaction->username}\n"
                . "Game: {$game->game_name}\n"
                . "Total Purchase: {$amountFormatted}\n"
                . "Task Number: {$transaction->invoice_id}\n"
                . "Payment Method: {$transaction->payment_method}\n"
                . "Platform URL: " . config('app.url');

            // Send poll and store message info
            $pollResponse = Telegram::sendPoll([
                'chat_id' => $this->chatId,
                'question' => $question,
                'options' => json_encode(['Accept', 'Declined']),
                'is_anonymous' => false,
            ]);

            // Update transaction with Telegram message details
            if ($transaction && isset($pollResponse['poll']['id'])) {
                $transaction->update([
                    'telegram_poll_id' => $pollResponse['poll']['id'],
                    'telegram_message_id' => $pollResponse['message_id'],
                    'telegram_chat_id' => $this->chatId
                ]);
                
                logEvent('info', 'WebhookController', 'Telegram poll sent and linked to transaction for payment Id ' . $data['id'], [
                    'transaction_id' => $transaction->id,
                    'telegram_message' => $pollResponse
                ]);
            }
        /*}
        else{
            logEvent('warning', 'WebhookController', 'Withdrawal decode failed', ['target_amount_paid' => $data['target_amount_paid'],'withdraw_request' => $withdraw_request], $userId);
        }*/

        return response()->json(['message' => 'Payment Paid Processed'], 200);
    }

    private function handlePaymentConfirmed($data)
    {
        $userId = $data['metadata']['user_id'] ?? null;

        logEvent('info', 'WebhookController', 'Processing payment.confirmed event for payment Id ' . $data['id'], ['payment_id' => $data['id']], $userId);

        $transaction = Transaction::where('transaction_id', $data['id'])->first();

        if ($transaction) {
            $transaction->update(['payment_status' => $data['status']]);
        }

        return response()->json(['message' => 'Payment Confirmed Processed'], 200);
    }

    private function handlePaymentExpired($data)
    {
        $userId = $data['metadata']['user_id'] ?? null;
        
        logEvent('warning', 'WebhookController', 'Processing payment.expired event for payment Id ' . $data['id'], ['payment_id' => $data['id']], $userId);

        Transaction::where('transaction_id', $data['id'])
            ->update(['payment_status' => $data['status']]);

        return response()->json(['message' => 'Payment Expired Processed'], 200);
    }

    private function handlePaymentCancelled($data)
    {
        $userId = $data['metadata']['user_id'] ?? null;

        logEvent('warning', 'WebhookController', 'Processing payment.cancelled event for payment Id ' . $data['id'], ['payment_id' => $data['id']], $userId);

        Transaction::where('transaction_id', $data['id'])
            ->update(['payment_status' => $data['status']]);

        return response()->json(['message' => 'Payment Cancelled Processed'], 200);
    }

    private function handleWithdrawCreated($data)
    {
        $userId = $data['metadata']['user_id'] ?? null;

        logEvent('info', 'WebhookController', 'Processing withdraw.created event for withdraw Id ' . $data['id'], ['withdraw_id' => $data['id']], $userId);

        //$amount = $data['exchange_rate'] * $data['target_amount_paid'];
        $amount = $data['amount'];

        Transaction::create([
            'user_id' => $userId,
            'game_id' => null,
            'amount' => $amount,
            'currency' => $data['currency'],
            'type' => 'withdrawal',
            'payment_status' => $data['status'],
            'transaction_id' => $data['id'],
            'payment_method' => $data['payment_method_paid_by'] ?? null,
        ]);

        //$this->updateWalletBalance($userId, -$amount);

        return response()->json(['message' => 'Withdrawal Created Processed'], 200);
    }

    private function handleWithdrawPaid($data)
    {
        logEvent('info', 'WebhookController', 'Processing withdraw.paid event for withdraw Id ' . $data['id'], ['withdraw_id' => $data['id']]);

        $transaction = Transaction::where('transaction_id', $data['id'])
            ->update(['payment_status' => $data['status']]);

        return response()->json(['message' => 'Withdrawal Paid Processed'], 200);
    }

    private function handleWithdrawFailed($data)
    {
        logEvent('error', 'WebhookController', 'Processing withdraw.failed event for withdraw Id ' . $data['id'], ['withdraw_id' => $data['id']]);

        $transaction = Transaction::where('transaction_id', $data['id'])->first();

        if ($transaction) {
            $transaction->update(['payment_status' => $data['status']]);
            //$this->updateWalletBalance($transaction->user_id, $transaction->amount);
        }

        return response()->json(['message' => 'Withdrawal Failed Processed'], 200);
    }

    private function updateWalletBalance($userId, $amount)
    {
        DB::transaction(function () use ($userId, $amount) {
            $wallet = Wallet::where('user_id', $userId)->firstOrCreate([
                'user_id' => $userId,
            ], [
                'balance' => 0
            ]);

            $wallet->increment('balance', $amount);
        });

        //logEvent('info', 'WebhookController', "User ID {$userId} wallet updated by {$amount}");
    }

    public function handleGGPayoutWebhook(Request $request)
    {
        $data = $request->all();
        $transaction = Transaction::where('mchOrderNo', $data['mchOrderNo'])->first();
        if (!$transaction) {
            return response('transaction not found', 404);
        }
        logEvent('info', 'WebhookController', 'Received GGPayout Webhook', $request->all());
        logEvent('info', 'WebhookController', 'GGPayout DB Transaction Details for order No ' . $data['orderNo'], $transaction);

        // Update transaction status based on payment state
        $statusMap = [
            0 => 'created',
            1 => 'pending',
            2 => 'completed',
            3 => 'failed',
            4 => 'revoked',
            5 => 'refunded',
            6 => 'closed'
        ];

        // Check if real amount exists and differs from order amount
        $realAmount = $data['realAmount'] ?? null;
        $orderAmount = $transaction->amount;
        
        if ($realAmount && $realAmount != $orderAmount) {
            // Handle different payment amount scenario
            $result = $this->paymentService->fillEcashappOrder(
                $data['orderNo'],
                $data['mchOrderNo'],
                $transaction->user_id
            );

            if (!$result) {
                logEvent('error', 'WebhookController', 'Failed to fill Ecashapp order', [
                    'orderNo' => $data['orderNo'],
                    'mchOrderNo' => $data['mchOrderNo'],
                    'realAmount' => $realAmount
                ]);
                return response('Failed to process order fill', 500);
            }

            $transaction->update([
                'amount' => $realAmount,
                //'real_amount' => $realAmount
            ]);
        }

        $transaction->update([
            'payment_status' => $statusMap[$data['state']] ?? 'unknown',
            'transaction_id' => $data['orderNo'],
            'processed_at' => now()
        ]);

        // Only send poll if transaction is completed (state = 2)
        if ($data['state'] == 2) {
            $IDNNo = User::find($transaction->user_id)->idn_no;
            $gameId = $transaction->game_id;
            $game = Game::find($gameId);
            $amountFormatted = number_format((float)($realAmount ?? $request->amount / 100), 2, '.', '');
            
            $question = "Player ID: {$IDNNo}\n"
                . "Player Nickname (username): {$transaction->username}\n"
                . "Game: {$game->game_name}\n"
                . "Total purchase: {$amountFormatted}\n"
                . "Task Number: {$transaction->invoice_id}\n"
                . "Payment Method: {$transaction->payment_method}\n"
                . "Platform URL: " . config('app.url');
            
            // Send poll and store message info
            $pollResponse = Telegram::sendPoll([
                'chat_id' => $this->chatId,
                'question' => $question,
                'options' => json_encode(['Accept', 'Declined']),
                'is_anonymous' => false,
            ]);
            
            // Update transaction with Telegram message details
            if ($transaction && isset($pollResponse['poll']['id'])) {
                $transaction->update([
                    'telegram_poll_id' => $pollResponse['poll']['id'],
                    'telegram_message_id' => $pollResponse['message_id'],
                    'telegram_chat_id' => $this->chatId
                ]);
                
                logEvent('info', 'WebhookController', 'Telegram poll sent and linked to transaction for order No ' . $data['orderNo'], [
                    'transaction_id' => $transaction->id,
                    'telegram_message' => $pollResponse
                ]);
            }
        }

        logEvent('info', 'WebhookController', 'GGPayout DB Transaction updated for order No ' . $data['orderNo'], [
            'status' => $statusMap[$data['state']],
            'realAmount' => $realAmount
        ]);

        return response('success', 200);
    }
}
