<?php

namespace App\Http\Controllers;

use App\Models\Game;
use App\Models\User;
use Inertia\Inertia;
use Illuminate\Http\Request;
use App\Services\PaymentService;
use Illuminate\Validation\ValidationException;
use App\Models\Transaction;
use Illuminate\Support\Facades\Auth;
use Telegram\Bot\Laravel\Facades\Telegram;
use App\Models\Setting;
use App\Models\WalletButton;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class PaymentController extends Controller
{
    protected $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    public function showPaymentForm()
    {

        /*$transaction = Transaction::where('transaction_id', $data['id'])->first();

        $decodedAddress = $this->paymentService->WithdrawalDecode('lnbc300n1p59qkvupp5hlgeefa5k9m3k4ly6kvrnllf9prvnrzc46yu96le8gdvayppq6tqdqqcqzzsxqyz5vqsp52jtx8e5e5tc6nj63jkk3xmpqn7r0l4h4qxvjwhy83eev6jvh3lns9qxpqysgqgejhq3avu2fqh632f6x9lzdarfcg8008shkn7u0vt6vua99g4knkvqm560tgw95wds6uadank0prc7yjp6v9q5w0pmzl6w77dzqpsfcqpra9dq');  
        dd($decodedAddress);*/
        
        // Fetch only active games for users
        $games = Game::active()->get();
        $settings = [
            'min_amount' => Setting::get('min_deposit_amount'),
            'max_amount' => Setting::get('max_deposit_amount'),
            'min_withdrawal_amount' => Setting::get('min_withdrawal_amount'),
            'max_withdrawal_amount' => Setting::get('max_withdrawal_amount'),
            'currencies' => json_decode(Setting::get('available_currencies')),
            'default_currency' => Setting::get('default_currency'),
            'enable_lightning' => Setting::get('enable_lightning_payment') === 'true',
            'enable_onchain' => Setting::get('enable_onchain_payment') === 'true',
            'enable_ggpayout' => Setting::get('enable_ggpayout') === 'true',
            'payment_gateway_notes' => Setting::get('payment_gateway_notes'),
            'bitcoin_gateway_notes' => Setting::get('bitcoin_gateway_notes'),
            'ggpayout_gateway_notes' => Setting::get('ggpayout_gateway_notes'),
            'payken_gateway_notes' => Setting::get('payken_gateway_notes'),
            'lightning_payment_note' => Setting::get('lightning_payment_note'),
            'usdt_gateway_notes' => Setting::get('usdt_gateway_notes'),
            'usdc_gateway_notes' => Setting::get('usdc_gateway_notes'),
            'enable_ethereum_payment' => Setting::get('enable_ethereum_payment') === 'true',
            'enable_tron_payment' => Setting::get('enable_tron_payment') === 'true',
            'enable_usdt_payment' => Setting::get('enable_usdt_payment') === 'true',
            'enable_usdc_payment' => Setting::get('enable_usdc_payment') === 'true',
            'enable_redeem' => Setting::get('enable_redeem') === 'true',
            'enable_payken' => Setting::get('enable_payken') === 'true',
            'bitcoin_fees_note' => sprintf(Setting::get('bitcoin_fees_note'), Setting::get('bitcoin_fees')),
            'bitcoin_fees' => Setting::get('bitcoin_fees'),
            'cashapp_fees_note' => sprintf(Setting::get('cashapp_fees_note'), Setting::get('cashapp_fees')),
            'cashapp_fees' => Setting::get('cashapp_fees'),
            'usdt_fees_note' => sprintf(Setting::get('usdt_fees_note'), Setting::get('usdt_fees')),
            'usdt_fees' => Setting::get('usdt_fees'),
            'usdc_fees_note' => sprintf(Setting::get('usdc_fees_note'), Setting::get('usdc_fees')),
            'usdc_fees' => Setting::get('usdc_fees'),
        ];

        // Get enabled wallet buttons and sort them by display order
        $walletButtons = WalletButton::where('is_enabled', true)
            ->orderBy('display_order')
            ->get()
            ->groupBy('type')
            ->toArray();
        
        return Inertia::render('Payment/Form', [
            'games' => $games,
            'settings' => $settings,
            'walletButtons' => $walletButtons
        ]);
    }

    public function generateBitcoinAddress(Request $request)
    {
        try {
            $minAmount = Setting::get('min_deposit_amount', 30);
            $maxAmount = Setting::get('max_deposit_amount', 500);

            $request->validate([
                'amount' => "required|numeric|min:$minAmount|max:$maxAmount",
                'game' => 'required|exists:games,id',
                'username' => 'required|string|max:255',
            ]);

            // Add USDT currency to request
            $request->merge(['currency' => 'SATS']);

            // Call the processPayment method from PaymentService for TrySpeed
            $paymentResponse = $this->paymentService->processTrySpeedPayment($request);
            //dd($paymentResponse);

            // Create or update transaction record
            $transaction = Transaction::updateOrCreate(
                [
                    'transaction_id' => $paymentResponse['id'],
                ],
                [
                    'user_id' => Auth::user()->id,
                    'username' => $request->input('username'),
                    'game_id' => $request->input('game'),
                    'amount' => $paymentResponse['amount'],
                    'target_amount' => $paymentResponse['target_amount'],
                    'exchange_rate' => $paymentResponse['exchange_rate'],
                    'currency' => $paymentResponse['currency'],
                    'type' => 'deposit',
                    'payment_status' => $paymentResponse['status'],
                    'payment_method' => $paymentResponse['payment_method'] ?? null,
                    'payment_gateway' => 'tryspeed'
                ]
            );

            // Prepare response data
            $responseData = [
                'paymentID' => $paymentResponse['id'],
                'TransactionData' => $transaction
            ];

            // Filter payment method options based on settings
            $paymentMethodOptions = ['paymentMethodOptions' => []];
            $availableOptions = $paymentResponse['payment_method_options'] ?? [];
            
            if (Setting::get('enable_lightning_payment') === 'true' && isset($availableOptions['lightning'])) {
                $paymentMethodOptions['paymentMethodOptions']['lightning'] = $availableOptions['lightning'];
            }
            
            if (Setting::get('enable_onchain_payment') === 'true' && isset($availableOptions['on_chain'])) {
                $paymentMethodOptions['paymentMethodOptions']['on_chain'] = $availableOptions['on_chain'];
            }

            // Add Ethereum payment method if available
            if (Setting::get('enable_ethereum_payment') === 'true' && isset($availableOptions['ethereum'])) {
                $paymentMethodOptions['paymentMethodOptions']['ethereum'] = $availableOptions['ethereum'];
            }

            // Add Tron payment method if available  
            if (Setting::get('enable_tron_payment') === 'true' && isset($availableOptions['tron'])) {
                $paymentMethodOptions['paymentMethodOptions']['tron'] = $availableOptions['tron'];
            }

            // If no payment methods are enabled, return with error
            if (empty($paymentMethodOptions['paymentMethodOptions'])) {
                throw new Exception('No payment methods are currently enabled. Please try again later.');
            }

            $responseData = array_merge($responseData, $paymentMethodOptions);

            // Get enabled wallet buttons for the response
            $walletButtons = WalletButton::where('is_enabled', true)
                ->orderBy('display_order')
                ->get()
                ->groupBy('type')
                ->toArray();
            
            $responseData['walletButtons'] = $walletButtons;
            
            return redirect()->route('payment.form')->with('data', $responseData);

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors());
        } catch (Exception $e) {
            \Log::error('TrySpeed payment processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }
    
    public function generateUSDTAddress(Request $request)
    {
        try {
            $minAmount = Setting::get('min_deposit_amount', 30);
            $maxAmount = Setting::get('max_deposit_amount', 500);

            $request->validate([
                'amount' => "required|numeric|min:$minAmount|max:$maxAmount",
                'game' => 'required|exists:games,id',
                'username' => 'required|string|max:255',
            ]);

            // Add USDT currency to request
            $request->merge(['currency' => 'USDT']);

            // Call the processPayment method from PaymentService for TrySpeed
            $paymentResponse = $this->paymentService->processTrySpeedPayment($request);

            // Create or update transaction record
            $transaction = Transaction::updateOrCreate(
                [
                    'transaction_id' => $paymentResponse['id'],
                ],
                [
                    'user_id' => Auth::user()->id,
                    'username' => $request->input('username'),
                    'game_id' => $request->input('game'),
                    'amount' => $paymentResponse['amount'],
                    'target_amount' => $paymentResponse['target_amount'],
                    'exchange_rate' => $paymentResponse['exchange_rate'],
                    'currency' => $paymentResponse['currency'],
                    'type' => 'deposit',
                    'payment_status' => $paymentResponse['status'],
                    'payment_method' => $paymentResponse['payment_method'] ?? null,
                    'payment_gateway' => 'tryspeed'
                ]
            );

            // Prepare response data
            $responseData = [
                'paymentID' => $paymentResponse['id'],
                'TransactionData' => $transaction
            ];

            // Filter payment method options based on settings
            $paymentMethodOptions = ['paymentMethodOptions' => []];
            $availableOptions = $paymentResponse['payment_method_options'] ?? [];
            
            if (Setting::get('enable_lightning_payment') === 'true' && isset($availableOptions['lightning'])) {
                $paymentMethodOptions['paymentMethodOptions']['lightning'] = $availableOptions['lightning'];
            }
            
            if (Setting::get('enable_onchain_payment') === 'true' && isset($availableOptions['on_chain'])) {
                $paymentMethodOptions['paymentMethodOptions']['on_chain'] = $availableOptions['on_chain'];
            }

            // Add Ethereum payment method if available
            if (Setting::get('enable_ethereum_payment') === 'true' && isset($availableOptions['ethereum'])) {
                $paymentMethodOptions['paymentMethodOptions']['ethereum'] = $availableOptions['ethereum'];
            }

            // Add Tron payment method if available  
            if (Setting::get('enable_tron_payment') === 'true' && isset($availableOptions['tron'])) {
                $paymentMethodOptions['paymentMethodOptions']['tron'] = $availableOptions['tron'];
            }

            // If no payment methods are enabled, return with error
            if (empty($paymentMethodOptions['paymentMethodOptions'])) {
                throw new Exception('No payment methods are currently enabled. Please try again later.');
            }

            $responseData = array_merge($responseData, $paymentMethodOptions);

            // Get enabled wallet buttons for the response
            $walletButtons = WalletButton::where('is_enabled', true)
                ->orderBy('display_order')
                ->get()
                ->groupBy('type')
                ->toArray();
            
            $responseData['walletButtons'] = $walletButtons;
            
            return redirect()->route('payment.form')->with('data', $responseData);

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors());
        } catch (Exception $e) {
            \Log::error('TrySpeed payment processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }


    public function generateUSDCAddress(Request $request)
    {
        try {
            $minAmount = Setting::get('min_deposit_amount', 30);
            $maxAmount = Setting::get('max_deposit_amount', 500);

            $request->validate([
                'amount' => "required|numeric|min:$minAmount|max:$maxAmount",
                'game' => 'required|exists:games,id',
                'username' => 'required|string|max:255',
            ]);

            // Add USDC currency to request
            $request->merge(['currency' => 'USDC']);

            // Call the processPayment method from PaymentService for TrySpeed
            $paymentResponse = $this->paymentService->processTrySpeedPayment($request);
            
            // Create or update transaction record
            $transaction = Transaction::updateOrCreate(
                [
                    'transaction_id' => $paymentResponse['id'],
                ],
                [
                    'user_id' => Auth::user()->id,
                    'username' => $request->input('username'),
                    'game_id' => $request->input('game'),
                    'amount' => $paymentResponse['amount'],
                    'target_amount' => $paymentResponse['target_amount'],
                    'exchange_rate' => $paymentResponse['exchange_rate'],
                    'currency' => $paymentResponse['currency'],
                    'type' => 'deposit',
                    'payment_status' => $paymentResponse['status'],
                    'payment_method' => $paymentResponse['payment_method'] ?? null,
                    'payment_gateway' => 'tryspeed'
                ]
            );

            // Prepare response data
            $responseData = [
                'paymentID' => $paymentResponse['id'],
                'TransactionData' => $transaction
            ];

            // Filter payment method options based on settings
            $paymentMethodOptions = ['paymentMethodOptions' => []];
            $availableOptions = $paymentResponse['payment_method_options'] ?? [];

            // Add Ethereum payment method if available
            if (Setting::get('enable_ethereum_payment') === 'true' && isset($availableOptions['ethereum'])) {
                $paymentMethodOptions['paymentMethodOptions']['ethereum'] = $availableOptions['ethereum'];
            }

            // If no payment methods are enabled, return with error
            if (empty($paymentMethodOptions['paymentMethodOptions'])) {
                throw new Exception('No payment methods are currently enabled. Please try again later.');
            }

            $responseData = array_merge($responseData, $paymentMethodOptions);

            // Get enabled wallet buttons for the response
            $walletButtons = WalletButton::where('is_enabled', true)
                ->orderBy('display_order')
                ->get()
                ->groupBy('type')
                ->toArray();
            
            $responseData['walletButtons'] = $walletButtons;
            
            return redirect()->route('payment.form')->with('data', $responseData);

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors());
        } catch (Exception $e) {
            \Log::error('TrySpeed payment processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function createGGPayoutOrder(Request $request)
    {
        try {
            $minAmount = Setting::get('min_deposit_amount', 30);
            $maxAmount = Setting::get('max_deposit_amount', 500);

            $request->validate([
                'amount' => "required|numeric|min:$minAmount|max:$maxAmount",
                'game' => 'required|exists:games,id',
                'username' => 'required|string|max:255',
                'wallet' => 'required|string'
            ]);

            $response = $this->paymentService->processGGPayoutPayment($request);

            if ($response['code'] === 0 && isset($response['payment_url'])) {
                // Create transaction record
                Transaction::create([
                    'user_id' => Auth::user()->id,
                    'username' => $request->input('username'),
                    'game_id' => $request->input('game'),
                    'amount' => $request->input('amount'),
                    'currency' => 'USD',
                    'type' => 'deposit',
                    'payment_status' => 'pending',
                    'mchOrderNo' => $response['id'] ?? null,
                    'payment_method' => $request->input('wallet'),
                    'payment_gateway' => 'ggpayout'
                ]);

                return Inertia::location($response['payment_url']);
            }
            
            throw new Exception($response['msg'] ?? 'Failed to create GGPayout order');

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors());
        } catch (Exception $e) {
            \Log::error('GGPayout order creation failed', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            /*return Inertia::render('Payment/GGPayoutError', [
                'message' => $e->getMessage()
            ]);*/
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    /*public function checkStatus($paymentId)
    {
        try {
            $transaction = Transaction::where('transaction_id', $paymentId)->first();

            if (!$transaction) {
                return response()->json(['status' => 'error', 'message' => 'Transaction not found'], 404);
            }

            // Check status based on payment gateway
            if ($transaction->payment_gateway === 'ggpayout') {
                $status = $this->paymentService->checkGGPayoutStatus($paymentId);
                if ($status) {
                    $transaction->update(['payment_status' => $status]);
                }
            } else if ($transaction->payment_gateway === 'tryspeed') {
                $status = $this->paymentService->checkTrySpeedStatus($paymentId);
                if ($status) {
                    $transaction->update(['payment_status' => $status]);
                }
            }

            return response()->json([
                'status' => $transaction->payment_status,
                'transaction_id' => $transaction->transaction_id,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
                'payment_method' => $transaction->payment_method,
                'payment_gateway' => $transaction->payment_gateway,
                'created_at' => $transaction->created_at->format('Y-m-d H:i:s'),
            ]);
        } catch (\Exception $e) {
            \Log::error('Payment status check error', [
                'error' => $e->getMessage(),
                'payment_id' => $paymentId
            ]);
            return response()->json(['status' => 'error', 'message' => 'Failed to check payment status'], 500);
        }
    }*/
    public function checkStatus($paymentId)
    {
        $transaction = Transaction::where('transaction_id', $paymentId)->first();

        if (!$transaction) {
            return response()->json(['status' => 'error', 'message' => 'Transaction not found'], 200);
        }

        return response()->json([
            'status' => $transaction->payment_status,
            'transaction_id' => $transaction->transaction_id,
            'amount' => $transaction->amount,
            'currency' => $transaction->currency,
            'payment_method' => $transaction->payment_method,
            'created_at' => $transaction->created_at->format('Y-m-d H:i:s'),
        ]);
    }

    public function removeOldTryspeedTransaction(Request $request) {
        Transaction::where('transaction_id', $request->input('paymentId'))->delete();
        return response()->json(['status' => 'success', 'message' => 'Transaction removed'], 200);
    }

    public function success($paymentId)
    {
        $transaction = Transaction::where('transaction_id', $paymentId)->firstOrFail();
        
        return inertia('Payment/PaymentSuccess', [
            'transaction' => $transaction
        ]);
    }

    public function ggpayoutSuccess(Request $request, $orderId)
    {
        // Retrieve the transaction based on the orderId
        $transaction = Transaction::where('mchOrderNo', $orderId)->firstOrFail();
        if($request->query('status') == 'failed'){
            $transaction->payment_status = $request->query('status');
            $transaction->save();
        }
        return inertia('Payment/Success', [
            'paymentId' => $orderId,
            'transaction' => $transaction,
            'status' => $request->query('status'),
            'mchOrderNo' => $request->query('mchOrderNo'),
            'amount' => $request->query('amount'),
            'auth' => [
                'user' => Auth::user()
            ]
        ]);
    }

    public function showRedeemForm()
    {
        // Fetch only active games for users
        $games = Game::active()->get();
        $settings = [
            'min_amount' => Setting::get('min_deposit_amount'),
            'max_amount' => Setting::get('max_deposit_amount'),
            'min_withdrawal_amount' => Setting::get('min_withdrawal_amount'),
            'max_withdrawal_amount' => Setting::get('max_withdrawal_amount'),
            'currencies' => json_decode(Setting::get('available_currencies')),
            'default_currency' => Setting::get('default_currency'),
            'enable_lightning' => Setting::get('enable_lightning_payment') === 'true',
            'enable_onchain' => Setting::get('enable_onchain_payment') === 'true',
            'enable_ggpayout' => Setting::get('enable_ggpayout') === 'true',
            'payment_gateway_notes' => Setting::get('payment_gateway_notes'),
            'bitcoin_gateway_notes' => Setting::get('bitcoin_gateway_notes'),
            'ggpayout_gateway_notes' => Setting::get('ggpayout_gateway_notes'),
            'payken_gateway_notes' => Setting::get('payken_gateway_notes'),
            'lightning_payment_note' => Setting::get('lightning_payment_note'),
            'usdt_gateway_notes' => Setting::get('usdt_gateway_notes'),
            'usdc_gateway_notes' => Setting::get('usdc_gateway_notes'),
            'enable_ethereum_payment' => Setting::get('enable_ethereum_payment') === 'true',
            'enable_tron_payment' => Setting::get('enable_tron_payment') === 'true',
            'enable_usdt_payment' => Setting::get('enable_usdt_payment') === 'true',
            'enable_usdc_payment' => Setting::get('enable_usdc_payment') === 'true',
            'enable_redeem' => Setting::get('enable_redeem') === 'true',
            'enable_payken' => Setting::get('enable_payken') === 'true',
            'bitcoin_fees_note' => sprintf(Setting::get('bitcoin_fees_note'), Setting::get('bitcoin_fees')),
            'bitcoin_fees' => Setting::get('bitcoin_fees'),
            'cashapp_fees_note' => sprintf(Setting::get('cashapp_fees_note'), Setting::get('cashapp_fees')),
            'cashapp_fees' => Setting::get('cashapp_fees'),
            'usdt_fees_note' => sprintf(Setting::get('usdt_fees_note'), Setting::get('usdt_fees')),
            'usdt_fees' => Setting::get('usdt_fees'),
            'usdc_fees_note' => sprintf(Setting::get('usdc_fees_note'), Setting::get('usdc_fees')),
            'usdc_fees' => Setting::get('usdc_fees'),
        ];
        
        return Inertia::render('Payment/RedeemForm', [
            'games' => $games,
            'settings' => $settings
        ]);
    }

    public function processRedeemRequest(Request $request)
    {
        try {
            $request->validate([
                'amount' => 'required|numeric|min:'.Setting::get('min_withdrawal_amount').'|max:'.Setting::get('max_withdrawal_amount'),
                'game' => 'required|exists:games,id',
                'username' => 'required|string|max:255',
                'network' => 'required|in:lightning,onchain,usdt',
                'address' => 'required|string',
            ]);

            $address = $request->input('address');
            if ($request->input('network') == 'lightning' && !str_starts_with($address, 'lnbc')) {
                return redirect()->back()->withErrors(['error' => 'Please check if you are using the correct address']);
            } elseif ($request->input('network') == 'onchain' && !str_starts_with($address, 'bc1')) {
                return redirect()->back()->withErrors(['error' => 'Please check if you are using the correct address']);
            }
            $targetCurrency = 'SATS';
            if($request->input('network') ==='usdt'){
                $targetCurrency = 'USDT';
            }
            // Check if address starts with 'lnbc' for lightning network invoices
            if ($request->input('network') === 'lightning') {
                if (!str_starts_with($request->input('address'), 'lnbc')) {
                    return redirect()->back()->withErrors(['error' => 'Please provide a valid Lightning Network invoice starting with lnbc']);
                }

                // Decode and validate the lightning invoice
                $decodedAddress = $this->paymentService->WithdrawalDecode($request->input('address'));
                if ($decodedAddress === false) {
                    return redirect()->back()->withErrors(['error' => 'Invalid Lightning Network invoice. Please ensure it has 0 amount set.']);
                }
            }

            // Create a withdrawal transaction
            $transaction = Transaction::create([
                'user_id' => Auth::id(),
                'username' => $request->input('username'),
                'game_id' => $request->input('game'),
                'amount' => $request->input('amount'),
                'currency' => 'USD',
                'target_currency' => $targetCurrency,
                'type' => 'withdrawal',
                'payment_status' => 'unpaid',
                'transaction_id' => null,
                'payment_method' => $request->input('network'),
                'payment_address' => $request->input('address')
            ]);

            // Send notification to Telegram
            $game = Game::find($request->input('game'));
            $user = Auth::user();
            
            // Format the message according to the specified format - without network and address
            $amount = number_format((float)$request->input('amount'), 2, '.', '');
            $message = "🚨🎰🎰🎰🚨\n\n"
                . "Task No.: " . $transaction->invoice_id . "\n"
                . "IDN No.: {$user->idn_no}\n"
                . "Player nickname: {$request->input('username')}\n"
                . "Game: {$game->game_name}\n"
                . "Redeem Points: {$amount}\n"
                . "Withdraw method: {$request->input('network')}";

            $telegramResponse = Telegram::sendMessage([
                'chat_id' => env('TELEGRAM_CHAT_ID_WITHDRAWAL'),
                'text' => $message
            ]);
            
            // Save the Telegram message ID for future reference
            if (isset($telegramResponse['message_id'])) {
                $transaction->update([
                    'telegram_message_id' => $telegramResponse['message_id'],
                    'telegram_chat_id' => env('TELEGRAM_CHAT_ID_WITHDRAWAL')
                ]);
            }

            return redirect()->route('redeem.success', ['transactionId' => $transaction->id]);
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors());
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function redeemSuccess($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);
        
        // Ensure the transaction belongs to the authenticated user
        if ($transaction->user_id != Auth::id()) {
            abort(403);
        }
        
        return Inertia::render('Payment/RedeemSuccess', [
            'transaction' => $transaction
        ]);
    }

    public function fillEcashappOrder(Request $request)
    {
        try {
            $validated = $request->validate([
                'payOrderNo' => 'required|string|max:30',
                'mchOrderNo' => 'required|string|max:30',
                'identifier' => ['required', 'string', 'regex:/^#.+$/'] // Must start with #
            ], [
                'payOrderNo.required' => 'Payment order number is required.',
                'mchOrderNo.required' => 'Merchant order number is required.',
                'identifier.required' => 'Transaction identifier is required.',
                'identifier.regex' => 'Transaction identifier must start with #.'
            ]);

            // Find the transaction
            $transaction = Transaction::where('mchOrderNo', $validated['mchOrderNo'])
                ->where('payment_method', 'ecashapp')
                ->firstOrFail();
            // Ensure the transaction belongs to the authenticated user
            if ($transaction->user_id != Auth::id()) {
                return back()->withErrors(['error' => 'Unauthorized access to transaction']);
            }

            // Call the payment service to fill the order
            $result = $this->paymentService->fillEcashappOrder(
                $validated['payOrderNo'],
                $validated['mchOrderNo'],
                $validated['identifier']
            );
            // Update transaction status
             // Update transaction status based on payment state
                $statusMap = [
                    0 => 'created',
                    1 => 'pending',
                    2 => 'completed',
                    3 => 'failed',
                    4 => 'revoked',
                    5 => 'refunded',
                    6 => 'closed'
                ];
            $transaction->update([
                'payment_status' => $statusMap[$result['state']] ?? 'failed',
                'transaction_id' => $result['transaction_id'],
                'processed_at' => now()
            ]);
            $user = User::find($transaction->user_id);
            $game = Game::find($transaction->game_id);
            $amountFormatted = number_format((float)$transaction->amount, 2, '.', '');
            
            $question = "Player ID: {$user->idn_no}\n"
                    . "Player Nickname (username): {$transaction->username}\n"
                    . "Game: {$game->game_name}\n"
                    . "Total purchase: {$amountFormatted}\n"
                    . "Task Number: {$transaction->mchOrderNo}\n"
                    . "Platform URL: " . config('app.url');
                
                // Send poll and store message info
                $pollResponse = Telegram::sendPoll([
                    'chat_id' => env('TELEGRAM_CHAT_ID_PAID'),
                    'question' => $question,
                    'options' => json_encode(['Accept', 'Declined']),
                    'is_anonymous' => false,
                ]);

                // Store Telegram message info in transaction
                if (isset($pollResponse['message_id'])) {
                    $transaction->update([
                        'telegram_message_id' => $pollResponse['message_id'],
                        'telegram_chat_id' => env('TELEGRAM_CHAT_ID_PAID')
                    ]);
                }

            return back()->with('success', $result['message']);

        } catch (ValidationException $e) {
            return back()->withErrors($e->errors());
        } catch (ModelNotFoundException $e) {
            return back()->withErrors(['error' => 'Transaction not found']);
        } catch (\Exception $e) {
            \Log::error('Ecashapp fill order failed', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function createPaykenOrder(Request $request)
    {
        try {
            $minAmount = Setting::get('min_deposit_amount', 30);
            $maxAmount = Setting::get('max_deposit_amount', 500);

            $request->validate([
                'amount' => "required|numeric|min:$minAmount|max:$maxAmount",
                'game' => 'required|exists:games,id',
                'username' => 'required|string|max:255'
            ]);

            $response = $this->paymentService->processPaykenPayment($request);

            if ($response['code'] === 0 && isset($response['payment_url'])) {
                return Inertia::location($response['payment_url']);
            }

            throw new Exception($response['msg'] ?? 'Failed to create Payken order');

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors());
        } catch (Exception $e) {
            \Log::error('Payken order creation failed', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            return redirect()->back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function handlePaykenCallback(Request $request)
    {
        logEvent('info', 'handlePaykenCallback', 'Received Webhook', $request->all());
        try {
            $response = $this->paymentService->handlePaykenCallback($request);
            
            if ($response['success']) {
                // Redirect to success page with transaction details
                $transaction = Transaction::where('mchOrderNo', $request->input('order_id'))->first();
                if ($transaction) {
                    return redirect()->route('payment.success', ['paymentId' => $transaction->transaction_id])
                        ->with('success', 'Payment processed successfully');
                }
            }

            return redirect()->route('payment.form')
                ->with('error', 'Failed to process payment');

        } catch (Exception $e) {
            \Log::error('Payken callback processing failed', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            return redirect()->route('payment.form')
                ->with('error', 'Failed to process payment callback');
        }
    }
}