<?php

namespace App\Http\Controllers;

use App\Models\Game;
use App\Models\Transaction; // Import the Transaction model
use Inertia\Inertia;
use App\Models\Setting;
use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Models\User;

class DashboardController extends Controller
{
    public function index()
    {
        // Fetch only active games for users, all games for admins
        $games = auth()->user()->role === 'admin' || auth()->user()->role === 'cashier' 
            ? Game::all() 
            : Game::active()->get();


        $settings = [
            'min_amount' => Setting::get('min_deposit_amount'),
            'max_amount' => Setting::get('max_deposit_amount'),
            'min_withdrawal_amount' => Setting::get('min_withdrawal_amount'),
            'max_withdrawal_amount' => Setting::get('max_withdrawal_amount'),
            'currencies' => json_decode(Setting::get('available_currencies')),
            'default_currency' => Setting::get('default_currency'),
            'enable_lightning' => Setting::get('enable_lightning_payment') === 'true',
            'enable_onchain' => Setting::get('enable_onchain_payment') === 'true',
            'enable_ggpayout' => Setting::get('enable_ggpayout') === 'true',
            'payment_gateway_notes' => Setting::get('payment_gateway_notes'),
            'bitcoin_gateway_notes' => Setting::get('bitcoin_gateway_notes'),
            'ggpayout_gateway_notes' => Setting::get('ggpayout_gateway_notes'),
            'payken_gateway_notes' => Setting::get('payken_gateway_notes'),
            'lightning_payment_note' => Setting::get('lightning_payment_note'),
            'usdt_gateway_notes' => Setting::get('usdt_gateway_notes'),
            'usdc_gateway_notes' => Setting::get('usdc_gateway_notes'),
            'enable_ethereum_payment' => Setting::get('enable_ethereum_payment') === 'true',
            'enable_tron_payment' => Setting::get('enable_tron_payment') === 'true',
            'enable_usdt_payment' => Setting::get('enable_usdt_payment') === 'true',
            'enable_usdc_payment' => Setting::get('enable_usdc_payment') === 'true',
            'enable_redeem' => Setting::get('enable_redeem') === 'true',
            'enable_payken' => Setting::get('enable_payken') === 'true',
            'bitcoin_fees_note' => sprintf(Setting::get('bitcoin_fees_note'), Setting::get('bitcoin_fees')),
            'bitcoin_fees' => Setting::get('bitcoin_fees'),
            'cashapp_fees_note' => sprintf(Setting::get('cashapp_fees_note'), Setting::get('cashapp_fees')),
            'cashapp_fees' => Setting::get('cashapp_fees'),
            'usdt_fees_note' => sprintf(Setting::get('usdt_fees_note'), Setting::get('usdt_fees')),
            'usdt_fees' => Setting::get('usdt_fees'),
            'usdc_fees_note' => sprintf(Setting::get('usdc_fees_note'), Setting::get('usdc_fees')),
            'usdc_fees' => Setting::get('usdc_fees'),
        ];

        // Fetch transaction statistics
        $today = now()->startOfDay();
        $last7Days = now()->subDays(7);
        $last30Days = now()->subDays(30);

        $totalDepositsToday = Transaction::whereDate('created_at', $today)->where('type', 'deposit')->whereIn('payment_status', ['paid', 'completed'])->count();
        $totalWithdrawalsToday = Transaction::whereDate('created_at', $today)->where('type', 'withdrawal')->whereIn('payment_status', ['paid', 'completed'])->count();
        
        $totalDepositsLast7Days = Transaction::where('created_at', '>=', $last7Days)->where('type', 'deposit')->whereIn('payment_status', ['paid', 'completed'])->count();
        $totalWithdrawalsLast7Days = Transaction::where('created_at', '>=', $last7Days)->where('type', 'withdrawal')->whereIn('payment_status', ['paid', 'completed'])->count();
        
        $totalDepositsLast30Days = Transaction::where('created_at', '>=', $last30Days)->where('type', 'deposit')->whereIn('payment_status', ['paid', 'completed'])->count();
        $totalWithdrawalsLast30Days = Transaction::where('created_at', '>=', $last30Days)->where('type', 'withdrawal')->whereIn('payment_status', ['paid', 'completed'])->count();

        // Redirect based on user role
        if (auth()->user()->role === 'admin' || auth()->user()->role === 'cashier') {
            return Inertia::render('Admin/Dashboard', [
                'games' => $games, // Pass games to the view
                'transactionStats' => [
                    'total_deposits_today' => $totalDepositsToday,
                    'total_deposits_last_7_days' => $totalDepositsLast7Days,
                    'total_deposits_last_30_days' => $totalDepositsLast30Days,
                    'total_withdrawals_today' => $totalWithdrawalsToday,
                    'total_withdrawals_last_7_days' => $totalWithdrawalsLast7Days,
                    'total_withdrawals_last_30_days' => $totalWithdrawalsLast30Days,
                ],
            ]); // Pass transaction stats to the admin dashboard
        }

        return Inertia::render('Dashboard', [
            'games' => $games, // Pass games to the view
            'settings' => $settings
        ]);
    }

    /**
     * Get transaction statistics by payment method with customizable time periods
     */
    public function getTransactionStats(Request $request)
    {
        // Set default period if not provided
        $period = $request->input('period', 'daily');
        
        // Validate the request
        $request->validate([
            'period' => 'in:daily,weekly,monthly,custom',
            'start_date' => 'required_if:period,custom|date',
            'end_date' => 'required_if:period,custom|date|after_or_equal:start_date',
        ]);

        $startDate = null;
        $endDate = null;

        // Set date range based on period
        switch ($period) {
            case 'daily':
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case 'weekly':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'monthly':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'custom':
                $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
                $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
                break;
        }

        // Get all unique payment methods from transactions
        $paymentMethods = Transaction::whereNotNull('payment_method')
            ->distinct()
            ->pluck('payment_method')
            ->toArray();

        $stats = [];
        $totalStats = [
            'total_amount' => 0,
            'total_transactions' => 0,
            'total_deposits' => 0,
            'total_withdrawals' => 0,
            'net_revenue' => 0,
            'avg_transaction_amount' => 0,
            'success_rate' => 0,
            'pending_transactions' => 0,
            'failed_transactions' => 0,
            'pending_amount' => 0,
            'failed_amount' => 0,
            'cancelled_transactions' => 0,
            'cancelled_amount' => 0
        ];

        foreach ($paymentMethods as $method) {
            // Get total amount processed for each payment method
            $totalAmount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['paid', 'completed'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            // Get transaction count for each payment method
            $transactionCount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['paid', 'completed'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

            // Get deposits and withdrawals separately
            $deposits = Transaction::where('payment_method', $method)
                ->where('type', 'deposit')
                ->whereIn('payment_status', ['paid', 'completed'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            $withdrawals = Transaction::where('payment_method', $method)
                ->where('type', 'withdrawal')
                ->whereIn('payment_status', ['paid', 'completed'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            // Get additional statistics
            $pendingCount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['pending', 'unpaid'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

            $pendingAmount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['pending', 'unpaid'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            $failedCount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['failed', 'expired'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

            $failedAmount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['failed', 'expired'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            $cancelledCount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['cancelled', 'closed'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

            $cancelledAmount = Transaction::where('payment_method', $method)
                ->whereIn('payment_status', ['cancelled', 'closed'])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount');

            $totalTransactionsForMethod = Transaction::where('payment_method', $method)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

            $successRate = $totalTransactionsForMethod > 0 ? 
                round(($transactionCount / $totalTransactionsForMethod) * 100, 2) : 0;

            $avgAmount = $transactionCount > 0 ? $totalAmount / $transactionCount : 0;

            $stats[$method] = [
                'total_amount' => $totalAmount,
                'transaction_count' => $transactionCount,
                'deposits' => $deposits,
                'withdrawals' => $withdrawals,
                'net_revenue' => $deposits - $withdrawals,
                'pending_count' => $pendingCount,
                'failed_count' => $failedCount,
                'cancelled_count' => $cancelledCount,
                'pending_amount' => $pendingAmount,
                'failed_amount' => $failedAmount,
                'cancelled_amount' => $cancelledAmount,
                'success_rate' => $successRate,
                'avg_amount' => $avgAmount,
                'formatted_total' => number_format($totalAmount, 2),
                'formatted_deposits' => number_format($deposits, 2),
                'formatted_withdrawals' => number_format($withdrawals, 2),
                'formatted_net_revenue' => number_format($deposits - $withdrawals, 2),
                'formatted_pending_amount' => number_format($pendingAmount, 2),
                'formatted_failed_amount' => number_format($failedAmount, 2),
                'formatted_cancelled_amount' => number_format($cancelledAmount, 2),
                'formatted_avg_amount' => number_format($avgAmount, 2),
            ];

            // Accumulate totals
            $totalStats['total_amount'] += $totalAmount;
            $totalStats['total_transactions'] += $transactionCount;
            $totalStats['total_deposits'] += $deposits;
            $totalStats['total_withdrawals'] += $withdrawals;
            $totalStats['net_revenue'] += ($deposits - $withdrawals);
            $totalStats['pending_transactions'] += $pendingCount;
            $totalStats['failed_transactions'] += $failedCount;
            $totalStats['cancelled_transactions'] += $cancelledCount;
            $totalStats['pending_amount'] += $pendingAmount;
            $totalStats['failed_amount'] += $failedAmount;
            $totalStats['cancelled_amount'] += $cancelledAmount;
        }

        // Calculate overall statistics
        $totalAllTransactions = Transaction::whereBetween('created_at', [$startDate, $endDate])->count();
        $totalStats['success_rate'] = $totalAllTransactions > 0 ? 
            round(($totalStats['total_transactions'] / $totalAllTransactions) * 100, 2) : 0;
        $totalStats['avg_transaction_amount'] = $totalStats['total_transactions'] > 0 ? 
            $totalStats['total_amount'] / $totalStats['total_transactions'] : 0;

        // Get daily breakdown for charts
        $dailyBreakdown = Transaction::whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, payment_method, SUM(amount) as total_amount, COUNT(*) as transaction_count')
            ->groupBy('date', 'payment_method')
            ->orderBy('date')
            ->get()
            ->groupBy('date');

        // Get top users by transaction volume
        $topUsers = Transaction::whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('user_id, username, SUM(amount) as total_amount, COUNT(*) as transaction_count')
            ->groupBy('user_id', 'username')
            ->orderByDesc('total_amount')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'period' => $period,
            'start_date' => $startDate->format('Y-m-d H:i:s'),
            'end_date' => $endDate->format('Y-m-d H:i:s'),
            'payment_methods' => $stats,
            'totals' => [
                'total_processed' => $totalStats['total_amount'],
                'total_transactions' => $totalStats['total_transactions'],
                'total_deposits' => $totalStats['total_deposits'],
                'total_withdrawals' => $totalStats['total_withdrawals'],
                'net_revenue' => $totalStats['net_revenue'],
                'pending_transactions' => $totalStats['pending_transactions'],
                'failed_transactions' => $totalStats['failed_transactions'],
                'cancelled_transactions' => $totalStats['cancelled_transactions'],
                'pending_amount' => $totalStats['pending_amount'],
                'failed_amount' => $totalStats['failed_amount'],
                'cancelled_amount' => $totalStats['cancelled_amount'],
                'success_rate' => $totalStats['success_rate'],
                'avg_transaction_amount' => $totalStats['avg_transaction_amount'],
                'formatted_total_processed' => number_format($totalStats['total_amount'], 2),
                'formatted_total_deposits' => number_format($totalStats['total_deposits'], 2),
                'formatted_total_withdrawals' => number_format($totalStats['total_withdrawals'], 2),
                'formatted_net_revenue' => number_format($totalStats['net_revenue'], 2),
                'formatted_pending_amount' => number_format($totalStats['pending_amount'], 2),
                'formatted_failed_amount' => number_format($totalStats['failed_amount'], 2),
                'formatted_cancelled_amount' => number_format($totalStats['cancelled_amount'], 2),
                'formatted_avg_amount' => number_format($totalStats['avg_transaction_amount'], 2),
            ],
            'daily_breakdown' => $dailyBreakdown,
            'top_users' => $topUsers,
            'summary' => [
                'total_payment_methods' => count($paymentMethods),
                'date_range_days' => $startDate->diffInDays($endDate) + 1,
                'period_label' => ucfirst($period),
            ]
        ]);
    }

    /**
     * Get period-over-period comparison statistics
     */
    public function getPeriodComparison(Request $request)
    {
        $request->validate([
            'current_period' => 'required|in:daily,weekly,monthly,custom',
            'compare_period' => 'required|in:daily,weekly,monthly,custom',
            'start_date' => 'required_if:current_period,custom|date',
            'end_date' => 'required_if:current_period,custom|date|after_or_equal:start_date',
            'compare_start_date' => 'required_if:compare_period,custom|date',
            'compare_end_date' => 'required_if:compare_period,custom|date|after_or_equal:compare_start_date',
        ]);

        $currentPeriod = $request->input('current_period');
        $comparePeriod = $request->input('compare_period');

        // Calculate current period stats
        $currentStats = $this->getPeriodStats($currentPeriod, false, $request);
        
        // Calculate comparison period stats (previous period of the same type)
        $compareStats = $this->getPeriodStats($currentPeriod, true, $request);

        // Calculate growth rates
        $growthRates = [];
        $metrics = ['total_deposits', 'total_withdrawals', 'net_revenue', 'total_transactions'];
        
        foreach ($metrics as $metric) {
            $current = $currentStats[$metric] ?? 0;
            $compare = $compareStats[$metric] ?? 0;
            
            if ($compare > 0) {
                $growthRate = (($current - $compare) / $compare) * 100;
            } else {
                $growthRate = $current > 0 ? 100 : 0;
            }
            
            $growthRates[$metric] = round($growthRate, 2);
        }

        return response()->json([
            'success' => true,
            'current_period' => $currentPeriod,
            'compare_period' => $comparePeriod,
            'current_stats' => $currentStats,
            'compare_stats' => $compareStats,
            'growth_rates' => $growthRates,
        ]);
    }

    /**
     * Helper method to get stats for a specific period
     */
    private function getPeriodStats($period, $isComparison = false, $request = null)
    {
        $startDate = null;
        $endDate = null;

        switch ($period) {
            case 'daily':
                if ($isComparison) {
                    $startDate = Carbon::now()->subDay()->startOfDay();
                    $endDate = Carbon::now()->subDay()->endOfDay();
                } else {
                    $startDate = Carbon::now()->startOfDay();
                    $endDate = Carbon::now()->endOfDay();
                }
                break;
            case 'weekly':
                if ($isComparison) {
                    $startDate = Carbon::now()->subWeek()->startOfWeek();
                    $endDate = Carbon::now()->subWeek()->endOfWeek();
                } else {
                    $startDate = Carbon::now()->startOfWeek();
                    $endDate = Carbon::now()->endOfWeek();
                }
                break;
            case 'monthly':
                if ($isComparison) {
                    $startDate = Carbon::now()->subMonth()->startOfMonth();
                    $endDate = Carbon::now()->subMonth()->endOfMonth();
                } else {
                    $startDate = Carbon::now()->startOfMonth();
                    $endDate = Carbon::now()->endOfMonth();
                }
                break;
            case 'custom':
                if ($isComparison) {
                    // For custom comparison, we need to calculate the previous period of the same duration
                    $currentStartDate = Carbon::parse($request->input('start_date'))->startOfDay();
                    $currentEndDate = Carbon::parse($request->input('end_date'))->endOfDay();
                    $duration = $currentStartDate->diffInDays($currentEndDate) + 1;
                    
                    $startDate = $currentStartDate->copy()->subDays($duration);
                    $endDate = $currentStartDate->copy()->subDay()->endOfDay();
                } else {
                    $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
                    $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
                }
                break;
        }

        $totalDeposits = Transaction::where('type', 'deposit')
            ->whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum('amount');

        $totalWithdrawals = Transaction::where('type', 'withdrawal')
            ->whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum('amount');

        $totalTransactions = Transaction::whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        return [
            'total_deposits' => $totalDeposits,
            'total_withdrawals' => $totalWithdrawals,
            'net_revenue' => $totalDeposits - $totalWithdrawals,
            'total_transactions' => $totalTransactions,
            'formatted_total_deposits' => number_format($totalDeposits, 2),
            'formatted_total_withdrawals' => number_format($totalWithdrawals, 2),
            'formatted_net_revenue' => number_format($totalDeposits - $totalWithdrawals, 2),
        ];
    }

    /**
     * Get customer transaction statistics with period-based filtering
     */
    public function getCustomerTransactionStats(Request $request)
    {
        // Set default period if not provided
        $period = $request->input('period', 'daily');
        
        // Validate the request
        $request->validate([
            'period' => 'in:daily,weekly,monthly,specific_date,custom',
            'start_date' => 'required_if:period,custom,specific_date|date',
            'end_date' => 'required_if:period,custom|date|after_or_equal:start_date',
            'customer_id' => 'nullable|integer|exists:users,id',
            'idn_no' => 'nullable|integer',
            'payment_status' => 'nullable|in:paid,completed,pending,unpaid,failed,expired,cancelled,closed',
            'transaction_type' => 'nullable|in:deposit,withdrawal',
            'payment_method' => 'nullable|string',
            'limit' => 'nullable|integer|min:1|max:100',
            'sort_by' => 'nullable|in:total_processed,transaction_count,last_transaction',
            'sort_order' => 'nullable|in:asc,desc',
        ]);

        $startDate = null;
        $endDate = null;

        // Set date range based on period
        switch ($period) {
            case 'daily':
                $startDate = Carbon::now()->startOfDay();
                $endDate = Carbon::now()->endOfDay();
                break;
            case 'weekly':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'monthly':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'specific_date':
                $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
                $endDate = Carbon::parse($request->input('start_date'))->endOfDay();
                break;
            case 'custom':
                $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
                $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
                break;
        }

        // Build base query with eager loading
        $query = Transaction::with(['user'])
            ->select([
                'user_id',
                \DB::raw('COUNT(*) as transaction_count'),
                \DB::raw('SUM(CASE WHEN type = "deposit" AND payment_status IN ("paid", "completed") THEN amount ELSE 0 END) as total_deposits'),
                \DB::raw('SUM(CASE WHEN type = "withdrawal" AND payment_status IN ("paid", "completed") THEN amount ELSE 0 END) as total_withdrawals'),
                \DB::raw('SUM(CASE WHEN payment_status IN ("paid", "completed") THEN amount ELSE 0 END) as total_processed'),
                \DB::raw('SUM(CASE WHEN payment_status IN ("pending", "unpaid") THEN amount ELSE 0 END) as total_pending'),
                \DB::raw('SUM(CASE WHEN payment_status IN ("failed", "expired") THEN amount ELSE 0 END) as total_failed'),
                \DB::raw('SUM(CASE WHEN payment_status IN ("cancelled", "closed") THEN amount ELSE 0 END) as total_cancelled'),
                \DB::raw('COUNT(CASE WHEN payment_status IN ("paid", "completed") THEN 1 END) as successful_transactions'),
                \DB::raw('COUNT(CASE WHEN payment_status IN ("pending", "unpaid") THEN 1 END) as pending_transactions'),
                \DB::raw('COUNT(CASE WHEN payment_status IN ("failed", "expired") THEN 1 END) as failed_transactions'),
                \DB::raw('COUNT(CASE WHEN payment_status IN ("cancelled", "closed") THEN 1 END) as cancelled_transactions'),
                \DB::raw('COUNT(CASE WHEN type = "deposit" THEN 1 END) as deposits_count'),
                \DB::raw('COUNT(CASE WHEN type = "withdrawal" THEN 1 END) as withdrawals_count'),
                \DB::raw('COUNT(CASE WHEN type = "deposit" AND payment_status IN ("paid", "completed") THEN 1 END) as processed_deposits_count'),
                \DB::raw('COUNT(CASE WHEN type = "withdrawal" AND payment_status IN ("paid", "completed") THEN 1 END) as processed_withdrawals_count'),
                \DB::raw('MAX(created_at) as last_transaction'),
                \DB::raw('MIN(created_at) as first_transaction'),
            ])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy('user_id');

        // Apply additional filters
        if ($request->filled('customer_id')) {
            $query->where('user_id', $request->input('customer_id'));
        }
        if ($request->filled('idn_no')) {
            $query->whereHas('user', function ($q) use ($request) {
                $q->where('idn_no', $request->input('idn_no'));
            });
        }
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->input('payment_status'));
        }
        if ($request->filled('transaction_type')) {
            $query->where('type', $request->input('transaction_type'));
        }
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->input('payment_method'));
        }

        // Apply sorting
        $sortBy = $request->input('sort_by', 'total_processed');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Apply limit
        $limit = $request->input('limit', 50);
        $customers = $query->limit($limit)->get();

        $customerStats = [];
        $totalStats = [
            'total_customers' => 0,
            'total_processed' => 0,
            'total_deposits' => 0,
            'total_withdrawals' => 0,
            'total_pending' => 0,
            'total_failed' => 0,
            'total_cancelled' => 0,
            'total_transactions' => 0,
            'successful_transactions' => 0,
            'pending_transactions' => 0,
            'failed_transactions' => 0,
            'cancelled_transactions' => 0,
            'deposits_count' => 0,
            'withdrawals_count' => 0,
            'processed_deposits_count' => 0,
            'processed_withdrawals_count' => 0,
        ];

        foreach ($customers as $customer) {
            // Get recent transactions for this customer
            $recentTransactions = Transaction::where('user_id', $customer->user_id)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get()
                ->map(function ($transaction) {
                    return [
                        'id' => $transaction->id,
                        'invoice_id' => $transaction->invoice_id,
                        'amount' => $transaction->amount,
                        'type' => $transaction->type,
                        'payment_method' => $transaction->payment_method ?: 'Unknown',
                        'payment_status' => $transaction->payment_status,
                        'created_at' => $transaction->created_at->format('Y-m-d H:i:s'),
                        'formatted_amount' => number_format($transaction->amount, 2),
                    ];
                });

            // Get payment method breakdown for this customer
            $paymentMethodBreakdown = Transaction::where('user_id', $customer->user_id)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->selectRaw('payment_method, COUNT(*) as count, SUM(amount) as total_amount')
                ->groupBy('payment_method')
                ->get()
                ->map(function ($method) {
                    return [
                        'payment_method' => $method->payment_method ?: 'Unknown',
                        'count' => $method->count,
                        'total_amount' => $method->total_amount,
                        'formatted_total_amount' => number_format($method->total_amount, 2),
                    ];
                });

            $netRevenue = $customer->total_deposits - $customer->total_withdrawals;
            $successRate = $customer->transaction_count > 0 ? 
                round(($customer->successful_transactions / $customer->transaction_count) * 100, 2) : 0;
            $avgTransactionAmount = $customer->transaction_count > 0 ? 
                $customer->total_processed / $customer->transaction_count : 0;

            $customerStats[] = [
                'user_id' => $customer->user_id,
                'username' => $customer->user->name ?? 'Unknown',
                'idn_no' => $customer->user->idn_no ?? null,
                'transaction_count' => $customer->transaction_count,
                'total_deposits' => $customer->total_deposits,
                'total_withdrawals' => $customer->total_withdrawals,
                'total_processed' => $customer->total_processed,
                'total_pending' => $customer->total_pending,
                'total_failed' => $customer->total_failed,
                'total_cancelled' => $customer->total_cancelled,
                'net_revenue' => $netRevenue,
                'successful_transactions' => $customer->successful_transactions,
                'pending_transactions' => $customer->pending_transactions,
                'failed_transactions' => $customer->failed_transactions,
                'cancelled_transactions' => $customer->cancelled_transactions,
                'deposits_count' => $customer->deposits_count,
                'withdrawals_count' => $customer->withdrawals_count,
                'processed_deposits_count' => $customer->processed_deposits_count,
                'processed_withdrawals_count' => $customer->processed_withdrawals_count,
                'success_rate' => $successRate,
                'avg_transaction_amount' => $avgTransactionAmount,
                'first_transaction' => $customer->first_transaction,
                'last_transaction' => $customer->last_transaction,
                'formatted_total_deposits' => number_format($customer->total_deposits, 2),
                'formatted_total_withdrawals' => number_format($customer->total_withdrawals, 2),
                'formatted_total_processed' => number_format($customer->total_processed, 2),
                'formatted_total_pending' => number_format($customer->total_pending, 2),
                'formatted_total_failed' => number_format($customer->total_failed, 2),
                'formatted_total_cancelled' => number_format($customer->total_cancelled, 2),
                'formatted_net_revenue' => number_format($netRevenue, 2),
                'formatted_avg_transaction_amount' => number_format($avgTransactionAmount, 2),
                'recent_transactions' => $recentTransactions,
                'payment_method_breakdown' => $paymentMethodBreakdown,
            ];

            // Accumulate totals
            $totalStats['total_customers']++;
            $totalStats['total_processed'] += $customer->total_processed;
            $totalStats['total_deposits'] += $customer->total_deposits;
            $totalStats['total_withdrawals'] += $customer->total_withdrawals;
            $totalStats['total_pending'] += $customer->total_pending;
            $totalStats['total_failed'] += $customer->total_failed;
            $totalStats['total_cancelled'] += $customer->total_cancelled;
            $totalStats['total_transactions'] += $customer->transaction_count;
            $totalStats['successful_transactions'] += $customer->successful_transactions;
            $totalStats['pending_transactions'] += $customer->pending_transactions;
            $totalStats['failed_transactions'] += $customer->failed_transactions;
            $totalStats['cancelled_transactions'] += $customer->cancelled_transactions;
            $totalStats['deposits_count'] += $customer->deposits_count;
            $totalStats['withdrawals_count'] += $customer->withdrawals_count;
            $totalStats['processed_deposits_count'] += $customer->processed_deposits_count;
            $totalStats['processed_withdrawals_count'] += $customer->processed_withdrawals_count;
        }

        // Calculate overall statistics
        $overallNetRevenue = $totalStats['total_deposits'] - $totalStats['total_withdrawals'];
        $overallSuccessRate = $totalStats['total_transactions'] > 0 ? 
            round(($totalStats['successful_transactions'] / $totalStats['total_transactions']) * 100, 2) : 0;
        $overallAvgTransactionAmount = $totalStats['total_transactions'] > 0 ? 
            $totalStats['total_processed'] / $totalStats['total_transactions'] : 0;

        // Get daily breakdown for charts
        $dailyBreakdown = Transaction::whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, user_id, username, SUM(amount) as total_amount, COUNT(*) as transaction_count')
            ->groupBy('date', 'user_id', 'username')
            ->orderBy('date')
            ->get()
            ->groupBy('date');

        // Get top customers by transaction volume
        $topCustomers = Transaction::whereIn('payment_status', ['paid', 'completed'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('user_id, username, SUM(amount) as total_amount, COUNT(*) as transaction_count')
            ->groupBy('user_id', 'username')
            ->orderByDesc('total_amount')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'period' => $period,
            'start_date' => $startDate->format('Y-m-d H:i:s'),
            'end_date' => $endDate->format('Y-m-d H:i:s'),
            'customers' => $customerStats,
            'totals' => [
                'total_customers' => $totalStats['total_customers'],
                'total_processed' => $totalStats['total_processed'],
                'total_deposits' => $totalStats['total_deposits'],
                'total_withdrawals' => $totalStats['total_withdrawals'],
                'total_pending' => $totalStats['total_pending'],
                'total_failed' => $totalStats['total_failed'],
                'total_cancelled' => $totalStats['total_cancelled'],
                'net_revenue' => $overallNetRevenue,
                'total_transactions' => $totalStats['total_transactions'],
                'successful_transactions' => $totalStats['successful_transactions'],
                'pending_transactions' => $totalStats['pending_transactions'],
                'failed_transactions' => $totalStats['failed_transactions'],
                'cancelled_transactions' => $totalStats['cancelled_transactions'],
                'deposits_count' => $totalStats['deposits_count'],
                'withdrawals_count' => $totalStats['withdrawals_count'],
                'processed_deposits_count' => $totalStats['processed_deposits_count'],
                'processed_withdrawals_count' => $totalStats['processed_withdrawals_count'],
                'success_rate' => $overallSuccessRate,
                'avg_transaction_amount' => $overallAvgTransactionAmount,
                'formatted_total_processed' => number_format($totalStats['total_processed'], 2),
                'formatted_total_deposits' => number_format($totalStats['total_deposits'], 2),
                'formatted_total_withdrawals' => number_format($totalStats['total_withdrawals'], 2),
                'formatted_total_pending' => number_format($totalStats['total_pending'], 2),
                'formatted_total_failed' => number_format($totalStats['total_failed'], 2),
                'formatted_total_cancelled' => number_format($totalStats['total_cancelled'], 2),
                'formatted_net_revenue' => number_format($overallNetRevenue, 2),
                'formatted_avg_transaction_amount' => number_format($overallAvgTransactionAmount, 2),
            ],
            'daily_breakdown' => $dailyBreakdown,
            'top_customers' => $topCustomers,
            'summary' => [
                'total_customers' => count($customerStats),
                'date_range_days' => $startDate->diffInDays($endDate) + 1,
                'period_label' => ucfirst($period),
            ]
        ]);
    }

    /**
     * Get detailed transaction history for a specific customer
     */
    public function getCustomerTransactionHistory(Request $request, $customerId)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'payment_status' => 'nullable|in:paid,completed,pending,unpaid,failed,expired,cancelled,closed',
            'transaction_type' => 'nullable|in:deposit,withdrawal',
            'payment_method' => 'nullable|string',
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:100',
            'sort_by' => 'nullable|in:created_at,amount,type,payment_status',
            'sort_order' => 'nullable|in:asc,desc',
        ]);

        // Get customer info first and validate
        $customer = User::find($customerId);
        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer not found'
            ], 404);
        }

        $query = Transaction::where('user_id', $customerId);

        // Apply date filters
        if ($request->filled('start_date')) {
            $query->where('created_at', '>=', Carbon::parse($request->input('start_date'))->startOfDay());
        }
        if ($request->filled('end_date')) {
            $query->where('created_at', '<=', Carbon::parse($request->input('end_date'))->endOfDay());
        }

        // Apply filters
        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->input('payment_status'));
        }
        if ($request->filled('transaction_type')) {
            $query->where('type', $request->input('transaction_type'));
        }
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->input('payment_method'));
        }

        // Apply sorting
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Get all transactions (no pagination on server)
        $transactions = $query->get();

        // Calculate comprehensive statistics
        $totalTransactions = $transactions->count();
        $totalAmount = $transactions->sum('amount');
        
        // Calculate comprehensive statistics from all transactions
        $totalTransactions = $transactions->count();
        $totalAmount = $transactions->sum('amount');
        
        // Deposit statistics
        $deposits = $transactions->where('type', 'deposit');
        $successfulDeposits = $deposits->whereIn('payment_status', ['paid', 'completed']);
        $pendingDeposits = $deposits->whereIn('payment_status', ['pending', 'unpaid']);
        $failedDeposits = $deposits->whereIn('payment_status', ['failed', 'expired', 'cancelled', 'closed']);
        
        $totalDeposits = $deposits->sum('amount');
        $successfulDepositsAmount = $successfulDeposits->sum('amount');
        $pendingDepositsAmount = $pendingDeposits->sum('amount');
        $failedDepositsAmount = $failedDeposits->sum('amount');
        
        // Withdrawal statistics
        $withdrawals = $transactions->where('type', 'withdrawal');
        $successfulWithdrawals = $withdrawals->whereIn('payment_status', ['paid', 'completed']);
        $pendingWithdrawals = $withdrawals->whereIn('payment_status', ['pending', 'unpaid']);
        $failedWithdrawals = $withdrawals->whereIn('payment_status', ['failed', 'expired', 'cancelled', 'closed']);
        
        $totalWithdrawals = $withdrawals->sum('amount');
        $successfulWithdrawalsAmount = $successfulWithdrawals->sum('amount');
        $pendingWithdrawalsAmount = $pendingWithdrawals->sum('amount');
        $failedWithdrawalsAmount = $failedWithdrawals->sum('amount');

        // Success rates
        $depositSuccessRate = $deposits->count() > 0 ? 
            round(($successfulDeposits->count() / $deposits->count()) * 100, 2) : 0;
        $withdrawalSuccessRate = $withdrawals->count() > 0 ? 
            round(($successfulWithdrawals->count() / $withdrawals->count()) * 100, 2) : 0;
        $overallSuccessRate = $totalTransactions > 0 ? 
            round((($successfulDeposits->count() + $successfulWithdrawals->count()) / $totalTransactions) * 100, 2) : 0;

        // Calculate net revenue (successful deposits - successful withdrawals)
        $netRevenue = $successfulDepositsAmount - $successfulWithdrawalsAmount;

        // Payment method breakdown - handle null/empty payment methods
        $paymentMethodStats = $transactions->groupBy(function ($transaction) {
            return $transaction->payment_method ?: 'Unknown';
        })->map(function ($methodTransactions) {
            $successful = $methodTransactions->whereIn('payment_status', ['paid', 'completed']);
            $total = $methodTransactions->count();
            $successRate = $total > 0 ? round(($successful->count() / $total) * 100, 2) : 0;
            
            return [
                'total_transactions' => $total,
                'successful_transactions' => $successful->count(),
                'total_amount' => $methodTransactions->sum('amount'),
                'successful_amount' => $successful->sum('amount'),
                'success_rate' => $successRate,
                'deposits' => $methodTransactions->where('type', 'deposit')->count(),
                'withdrawals' => $methodTransactions->where('type', 'withdrawal')->count(),
                'deposits_amount' => $methodTransactions->where('type', 'deposit')->sum('amount'),
                'withdrawals_amount' => $methodTransactions->where('type', 'withdrawal')->sum('amount'),
            ];
        });

        // Monthly breakdown
        $monthlyStats = $transactions->groupBy(function ($transaction) {
            return $transaction->created_at->format('Y-m');
        })->map(function ($monthTransactions) {
            $successful = $monthTransactions->whereIn('payment_status', ['paid', 'completed']);
            $deposits = $monthTransactions->where('type', 'deposit');
            $withdrawals = $monthTransactions->where('type', 'withdrawal');
            
            return [
                'total_transactions' => $monthTransactions->count(),
                'successful_transactions' => $successful->count(),
                'total_amount' => $monthTransactions->sum('amount'),
                'successful_amount' => $successful->sum('amount'),
                'deposits_count' => $deposits->count(),
                'withdrawals_count' => $withdrawals->count(),
                'deposits_amount' => $deposits->sum('amount'),
                'withdrawals_amount' => $withdrawals->sum('amount'),
                'success_rate' => $monthTransactions->count() > 0 ? 
                    round(($successful->count() / $monthTransactions->count()) * 100, 2) : 0,
            ];
        });

        return response()->json([
            'success' => true,
            'customer' => [
                'id' => $customer->id,
                'name' => $customer->name,
                'email' => $customer->email,
                'role' => $customer->role,
                'idn_no' => $customer->idn_no,
            ],
            'filters' => [
                'start_date' => $request->input('start_date'),
                'end_date' => $request->input('end_date'),
                'payment_status' => $request->input('payment_status'),
                'transaction_type' => $request->input('transaction_type'),
                'payment_method' => $request->input('payment_method'),
                'sort_by' => $sortBy,
                'sort_order' => $sortOrder,
            ],
            'summary' => [
                'total_transactions' => $totalTransactions,
                'total_amount' => $totalAmount,
                'formatted_total_amount' => number_format($totalAmount, 2),
                'overall_success_rate' => $overallSuccessRate,
                'net_revenue' => $netRevenue,
                'formatted_net_revenue' => number_format($netRevenue, 2),
                'date_range' => $request->filled('start_date') && $request->filled('end_date') ? 
                    Carbon::parse($request->input('start_date'))->format('M d, Y') . ' - ' . 
                    Carbon::parse($request->input('end_date'))->format('M d, Y') : 'All time',
            ],
            'deposits' => [
                'total_count' => $deposits->count(),
                'successful_count' => $successfulDeposits->count(),
                'pending_count' => $pendingDeposits->count(),
                'failed_count' => $failedDeposits->count(),
                'total_amount' => $totalDeposits,
                'successful_amount' => $successfulDepositsAmount,
                'pending_amount' => $pendingDepositsAmount,
                'failed_amount' => $failedDepositsAmount,
                'success_rate' => $depositSuccessRate,
                'formatted_total_amount' => number_format($totalDeposits, 2),
                'formatted_successful_amount' => number_format($successfulDepositsAmount, 2),
                'formatted_pending_amount' => number_format($pendingDepositsAmount, 2),
                'formatted_failed_amount' => number_format($failedDepositsAmount, 2),
            ],
            'withdrawals' => [
                'total_count' => $withdrawals->count(),
                'successful_count' => $successfulWithdrawals->count(),
                'pending_count' => $pendingWithdrawals->count(),
                'failed_count' => $failedWithdrawals->count(),
                'total_amount' => $totalWithdrawals,
                'successful_amount' => $successfulWithdrawalsAmount,
                'pending_amount' => $pendingWithdrawalsAmount,
                'failed_amount' => $failedWithdrawalsAmount,
                'success_rate' => $withdrawalSuccessRate,
                'formatted_total_amount' => number_format($totalWithdrawals, 2),
                'formatted_successful_amount' => number_format($successfulWithdrawalsAmount, 2),
                'formatted_pending_amount' => number_format($pendingWithdrawalsAmount, 2),
                'formatted_failed_amount' => number_format($failedWithdrawalsAmount, 2),
            ],
            'payment_methods' => $paymentMethodStats,
            'monthly_breakdown' => $monthlyStats,
            'transactions' => $transactions->map(function ($transaction) {
                return [
                    'id' => $transaction->id,
                    'invoice_id' => $transaction->invoice_id,
                    'amount' => $transaction->amount,
                    'type' => $transaction->type,
                    'payment_method' => $transaction->payment_method ?: 'Unknown',
                    'payment_status' => $transaction->payment_status,
                    'payment_gateway' => $transaction->payment_gateway,
                    'created_at' => $transaction->created_at->format('Y-m-d H:i:s'),
                    'processed_at' => $transaction->processed_at ? $transaction->processed_at->format('Y-m-d H:i:s') : null,
                    'formatted_amount' => number_format($transaction->amount, 2),
                    'is_successful' => in_array($transaction->payment_status, ['paid', 'completed']),
                    'is_pending' => in_array($transaction->payment_status, ['pending', 'unpaid']),
                    'is_failed' => in_array($transaction->payment_status, ['failed', 'expired', 'cancelled', 'closed']),
                ];
            }),
            'total_transactions' => $transactions->count(),
        ]);
    }

    /**
     * Show customer transaction history page
     */
    public function showCustomerTransactionHistory($customerId)
    {
        $customer = User::findOrFail($customerId);
        
        return Inertia::render('Admin/CustomerTransactionHistory', [
            'customer' => [
                'id' => $customer->id,
                'name' => $customer->name,
                'email' => $customer->email,
                'role' => $customer->role,
                'idn_no' => $customer->idn_no,
            ]
        ]);
    }

    /**
     * Download customer transaction report as CSV
     */
    public function downloadCustomerTransactionReport($customerId)
    {
        try {
            $customer = User::findOrFail($customerId);
            
            // Get all transactions for the customer
            $transactions = Transaction::where('user_id', $customerId)
                ->orderBy('created_at', 'desc')
                ->get();

            // Calculate comprehensive statistics
            $deposits = $transactions->where('type', 'deposit');
            $withdrawals = $transactions->where('type', 'withdrawal');
            
            $totalDeposits = $deposits->sum('amount');
            $successfulDeposits = $deposits->whereIn('payment_status', ['paid', 'completed'])->sum('amount');
            $pendingDeposits = $deposits->whereIn('payment_status', ['pending', 'unpaid'])->sum('amount');
            $failedDeposits = $deposits->whereIn('payment_status', ['failed', 'expired', 'cancelled', 'closed'])->sum('amount');
            
            $totalWithdrawals = $withdrawals->sum('amount');
            $successfulWithdrawals = $withdrawals->whereIn('payment_status', ['paid', 'completed'])->sum('amount');
            $pendingWithdrawals = $withdrawals->whereIn('payment_status', ['pending', 'unpaid'])->sum('amount');
            $failedWithdrawals = $withdrawals->whereIn('payment_status', ['failed', 'expired', 'cancelled', 'closed'])->sum('amount');
            
            $depositSuccessRate = $totalDeposits > 0 ? ($successfulDeposits / $totalDeposits) * 100 : 0;
            $withdrawalSuccessRate = $totalWithdrawals > 0 ? ($successfulWithdrawals / $totalWithdrawals) * 100 : 0;
            
            $netRevenue = $successfulDeposits - $successfulWithdrawals;
            $totalTransactions = $transactions->count();

            // Create CSV content
            $customerName = preg_replace('/[^a-zA-Z0-9]/', '_', $customer->name); // Remove special characters
            $filename = "transaction_report_{$customer->idn_no}_{$customerName}_" . date('Y-m-d_H-i-s') . ".csv";
            
            $headers = [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => "attachment; filename=\"$filename\"",
                'Pragma' => 'no-cache',
                'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
                'Expires' => '0'
            ];

            $callback = function() use ($transactions, $customer, $totalDeposits, $successfulDeposits, $pendingDeposits, $failedDeposits, $totalWithdrawals, $successfulWithdrawals, $pendingWithdrawals, $failedWithdrawals, $depositSuccessRate, $withdrawalSuccessRate, $netRevenue, $totalTransactions) {
                $file = fopen('php://output', 'w');
                
                // Add BOM for UTF-8
                fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
                
                // ===== REPORT HEADER =====
                fputcsv($file, ['VEGAS FISH - CUSTOMER TRANSACTION REPORT']);
                fputcsv($file, ['']);
                
                // ===== CUSTOMER INFORMATION =====
                fputcsv($file, ['CUSTOMER INFORMATION']);
                fputcsv($file, ['Customer Name', $customer->name]);
                fputcsv($file, ['IDN Number', $customer->idn_no]);
                fputcsv($file, ['Email Address', $customer->email ?? 'N/A']);
                fputcsv($file, ['Account Role', ucfirst($customer->role)]);
                fputcsv($file, ['Report Generated', now()->format('F d, Y \a\t g:i A')]);
                fputcsv($file, ['']);
                
                // ===== EXECUTIVE SUMMARY =====
                fputcsv($file, ['EXECUTIVE SUMMARY']);
                fputcsv($file, ['Total Transactions', $totalTransactions]);
                fputcsv($file, ['Net Revenue (Successful Deposits - Successful Withdrawals)', '$' . number_format($netRevenue, 2)]);
                fputcsv($file, ['']);
                
                // ===== DEPOSIT SUMMARY =====
                fputcsv($file, ['DEPOSIT SUMMARY']);
                fputcsv($file, ['Total Deposit Amount', '$' . number_format($totalDeposits, 2)]);
                fputcsv($file, ['Successful Deposits', '$' . number_format($successfulDeposits, 2)]);
                fputcsv($file, ['Pending Deposits', '$' . number_format($pendingDeposits, 2)]);
                fputcsv($file, ['Failed/Expired Deposits', '$' . number_format($failedDeposits, 2)]);
                fputcsv($file, ['Deposit Success Rate', number_format($depositSuccessRate, 1) . '%']);
                fputcsv($file, ['']);
                
                // ===== WITHDRAWAL SUMMARY =====
                fputcsv($file, ['WITHDRAWAL SUMMARY']);
                fputcsv($file, ['Total Withdrawal Amount', '$' . number_format($totalWithdrawals, 2)]);
                fputcsv($file, ['Successful Withdrawals', '$' . number_format($successfulWithdrawals, 2)]);
                fputcsv($file, ['Pending Withdrawals', '$' . number_format($pendingWithdrawals, 2)]);
                fputcsv($file, ['Failed/Expired Withdrawals', '$' . number_format($failedWithdrawals, 2)]);
                fputcsv($file, ['Withdrawal Success Rate', number_format($withdrawalSuccessRate, 1) . '%']);
                fputcsv($file, ['']);
                
                // ===== TRANSACTION DETAILS =====
                fputcsv($file, ['TRANSACTION DETAILS']);
                fputcsv($file, [
                    'Transaction ID',
                    'Invoice ID',
                    'Transaction Type',
                    'Amount ($)',
                    'Currency',
                    'Target Amount',
                    'Target Currency',
                    'Exchange Rate',
                    'Payment Method',
                    'Payment Status',
                    'Payment Gateway',
                    'Transaction ID (Gateway)',
                    'Merchant Order No',
                    'Payment Address',
                    'Username',
                    'Game Name',
                ]);
                
                // Add transaction data
                foreach ($transactions as $transaction) {
                    $game = Game::find($transaction->game_id);
                    $gameName = $game ? $game->game_name : 'N/A';
                    
                    fputcsv($file, [
                        $transaction->id,
                        $transaction->invoice_id ?? 'N/A',
                        ucfirst($transaction->type),
                        number_format($transaction->amount, 2),
                        $transaction->currency ?? 'USD',
                        $transaction->target_amount ? number_format($transaction->target_amount, 2) : 'N/A',
                        $transaction->target_currency ?? 'N/A',
                        $transaction->exchange_rate ?? 'N/A',
                        $transaction->payment_method ?? 'N/A',
                        ucfirst($transaction->payment_status),
                        $transaction->payment_gateway ?? 'N/A',
                        $transaction->transaction_id ?? 'N/A',
                        $transaction->mchOrderNo ?? 'N/A',
                        $transaction->payment_address ?? 'N/A',
                        $transaction->username ?? 'N/A',
                        $gameName
                    ]);
                }
                
                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to generate report. Please try again.');
        }
    }
} 