<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WalletButton;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Storage;

class WalletButtonController extends Controller
{
    public function index()
    {
        $walletButtons = WalletButton::orderBy('display_order')->get();
        return Inertia::render('Admin/WalletButtons/Index', [
            'walletButtons' => $walletButtons
        ]);
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'icon_url' => 'required|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                'wallet_url' => 'required|string|max:255',
                'type' => 'required|in:lightning,onchain,both',
                'is_enabled' => 'boolean',
                'display_order' => 'required|integer',
                //'note' => 'nullable|string|max:255'
            ]);

            if ($request->hasFile('icon_url')) {
                $iconPath = $request->file('icon_url')->store('wallet-icons', 'public');
                $validated['icon_url'] = $iconPath;
            }

            $walletButton = WalletButton::create($validated);

            return redirect()->back()->with([
                'success' => true,
                'message' => 'Wallet button created successfully',
                'wallet_button' => $walletButton
            ]);
        } catch (\Exception $e) {
            \Log::error('Wallet button creation failed: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to create wallet button']);
        }
    }

    public function update(Request $request, WalletButton $walletButton)
    {
        try {
            \Log::info('Update WalletButton Request:', [
                'request_all' => $request->all(),
                'files' => $request->allFiles(),
                'has_file' => $request->hasFile('icon_url')
            ]);

            // Validate the request
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'wallet_url' => 'required|string|max:255',
                'type' => 'required|in:lightning,onchain,both',
                'is_enabled' => 'required|boolean',
                'display_order' => 'required|integer',
                'icon_url' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                //'note' => 'nullable|string|max:255'
            ]);

            // Convert is_enabled to boolean
            $validated['is_enabled'] = filter_var($validated['is_enabled'], FILTER_VALIDATE_BOOLEAN);

            // Handle file upload if present
            if ($request->hasFile('icon_url')) {
                // Delete old icon if it exists
                if ($walletButton->icon_url) {
                    Storage::disk('public')->delete($walletButton->icon_url);
                }
                
                // Store new icon
                $iconPath = $request->file('icon_url')->store('wallet-icons', 'public');
                $validated['icon_url'] = $iconPath;
            } else {
                // Remove icon_url from validated data if no new file was uploaded
                unset($validated['icon_url']);
            }

            // Update the wallet button
            $walletButton->update($validated);

            // Get the updated list of buttons
            $walletButtons = WalletButton::orderBy('display_order')->get();

            // Redirect back to index with success message
            return redirect()->route('admin.wallet-buttons.index')->with([
                'success' => true,
                'message' => 'Wallet button updated successfully',
                'walletButtons' => $walletButtons
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation failed:', ['errors' => $e->errors()]);
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            \Log::error('Wallet button update failed: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to update wallet button']);
        }
    }

    public function toggleStatus(WalletButton $walletButton)
    {
        try {
            $walletButton->update([
                'is_enabled' => !$walletButton->is_enabled
            ]);

            return back()->with([
                'success' => true,
                'message' => 'Status updated successfully'
            ]);
        } catch (\Exception $e) {
            \Log::error('Status toggle failed: ' . $e->getMessage());
            return back()->withErrors([
                'error' => 'Failed to toggle status'
            ])->with([
                'success' => false,
                'message' => 'Failed to toggle status'
            ]);
        }
    }

    public function destroy(WalletButton $walletButton)
    {
        $walletButton->delete();

        return redirect()->back()->with('success', 'Wallet button deleted successfully.');
    }

    public function updateOrder(Request $request)
    {
        try {
            $request->validate([
                'buttons' => 'required|string'
            ]);

            // Decode the JSON string
            $buttons = json_decode($request->buttons, true);

            // Validate the decoded data
            if (!is_array($buttons)) {
                throw new \Exception('Invalid buttons data format');
            }

            foreach ($buttons as $button) {
                if (!isset($button['id']) || !isset($button['display_order'])) {
                    throw new \Exception('Missing required fields in button data');
                }
            }

            // Update each button's display order
            foreach ($buttons as $button) {
                WalletButton::where('id', $button['id'])
                    ->update(['display_order' => $button['display_order']]);
            }

            // Get the updated list of buttons
            $updatedButtons = WalletButton::orderBy('display_order')->get();

            return back()->with([
                'success' => true,
                'message' => 'Order updated successfully',
                'all_buttons' => $updatedButtons
            ]);
        } catch (\Exception $e) {
            \Log::error('Order update failed: ' . $e->getMessage(), [
                'request_data' => $request->all()
            ]);
            return back()->withErrors(['error' => 'Failed to update order'])->with([
                'success' => false,
                'message' => 'Failed to update order'
            ]);
        }
    }
}