<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Illuminate\Validation\Rules\Password;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $query = User::query()
            ->when($request->search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('idn_no', 'like', "%{$search}%");
                });
            })
            ->when($request->role, function ($query, $role) {
                $query->where('role', $role);
            })
            ->when($request->status, function ($query, $status) {
                $query->where('status', $status);
            });

        $users = $query->latest()->paginate(10)->withQueryString();

        // Map users to include status explicitly
        $users->getCollection()->transform(function ($user) {
            return [
                'id' => $user->id,
                'name' => $user->name,
                'idn_no' => $user->idn_no,
                'role' => $user->role,
                'status' => $user->status,
                'created_at' => $user->created_at,
            ];
        });

        return Inertia::render('Admin/Users/Index', [
            'users' => $users,
            'filters' => $request->only(['search', 'role', 'status'])
        ]);
    }

    public function create()
    {
        do {
            $idn_no = rand(1000000, 9999999); // Generate a random integer for idn_no
        } while (User::where('idn_no', $idn_no)->exists()); // Check for uniqueness

        return Inertia::render('Admin/Users/Create', [
            'idn_no' => $idn_no,
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'idn_no' => 'required|max:50|unique:users',
            'password' => ['required', Password::defaults()],
            'role' => 'required|in:customer,cashier,admin',
        ]);

        User::create([
            'name' => $request->name,
            'idn_no' => $request->idn_no,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'status' => 'active',
        ]);

        return redirect()->route('admin.users.index')
            ->with('success', 'User created successfully.');
    }

    public function edit(User $user)
    {
        return Inertia::render('Admin/Users/Edit', [
            'user' => $user
        ]);
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'idn_no' => 'required|max:50|unique:users,idn_no,' . $user->id,
            'role' => 'required|in:customer,cashier,admin',
            'status' => 'required|in:active,banned',
            'password' => ['nullable', Password::defaults()],
        ]);

        $userData = [
            'name' => $request->name,
            'idn_no' => $request->idn_no,
            'role' => $request->role,
            'status' => $request->status,
        ];

        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->password);
        }

        $user->update($userData);

        return redirect()->route('admin.users.index')
            ->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return back()->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return back()->with('success', 'User deleted successfully.');
    }

    public function toggleStatus(User $user)
    {
        if ($user->id === auth()->id()) {
            return redirect()->back()->with('error', 'You cannot change your own status.');
        }
        $user->toggleStatus();
        return redirect()->route('admin.users.index')->with('success', 'User status updated.');
    }
} 