<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $query = Transaction::with(['user', 'game'])
            ->when($request->search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('username', 'like', "%{$search}%")
                      ->orWhere('invoice_id', 'like', "%{$search}%")
                      ->orWhere('transaction_id', 'like', "%{$search}%")
                      ->orWhere('amount', 'like', "%{$search}%")
                      ->orWhereHas('user', function ($userQuery) use ($search) {
                          $userQuery->where('name', 'like', "%{$search}%");
                      });
                });
            })
            ->when($request->type, function ($query, $type) {
                $query->where('type', $type);
            })
            ->when($request->status, function ($query, $status) {
                $query->where('payment_status', $status);
            });

        $transactions = $query->latest()->paginate(10)->withQueryString();

        return Inertia::render('Admin/Transactions/Index', [
            'transactions' => $transactions,
            'filters' => $request->only(['search', 'type', 'status'])
        ]);
    }

    public function show(Transaction $transaction)
    {
        $transaction->load(['user', 'game']);
        
        return Inertia::render('Admin/Transactions/Show', [
            'transaction' => $transaction
        ]);
    }

    public function updateStatus(Transaction $transaction, Request $request)
    {
        $request->validate([
            'status' => 'required|in:unpaid,paid,expired,cancelled'
        ]);

        $transaction->update([
            'payment_status' => $request->status
        ]);

        return redirect()->back()->with('success', 'Transaction status updated successfully');
    }
} 