<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Promo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class PromoController extends Controller
{
    public function index()
    {
        $promos = Promo::orderBy('display_order')
            ->orderBy('created_at', 'desc')
            ->get();
        return Inertia::render('Admin/Promos/Index', [
            'promos' => $promos,
        ]);
    }

    public function create()
    {
        return Inertia::render('Admin/Promos/Create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'image_alt' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'display_order' => 'integer|min:0',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        // Handle image upload
        if ($request->hasFile('image') && $request->file('image')->isValid()) {
            $imagePath = $request->file('image')->store('promos', 'public');
            $validated['image_path'] = $imagePath;
        }

        Promo::create($validated);

        return redirect()->route('admin.promos.index')
            ->with('success', 'Promo created successfully');
    }

    public function show(Promo $promo)
    {
        return Inertia::render('Admin/Promos/Show', [
            'promo' => $promo,
        ]);
    }

    public function edit(Promo $promo)
    {
        return Inertia::render('Admin/Promos/Edit', [
            'promo' => $promo,
        ]);
    }

    public function update(Request $request, Promo $promo)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'image_alt' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'display_order' => 'integer|min:0',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        // Handle image upload
        if ($request->hasFile('image') && $request->file('image')->isValid()) {
            // Delete old image
            if ($promo->image_path) {
                Storage::disk('public')->delete($promo->image_path);
            }
            
            $imagePath = $request->file('image')->store('promos', 'public');
            $validated['image_path'] = $imagePath;
        }

        $promo->update($validated);

        return redirect()->route('admin.promos.index')
            ->with('success', 'Promo updated successfully');
    }

    public function destroy(Promo $promo)
    {
        // Delete image file
        if ($promo->image_path) {
            Storage::disk('public')->delete($promo->image_path);
        }

        $promo->delete();

        return redirect()->route('admin.promos.index')
            ->with('success', 'Promo deleted successfully');
    }
} 