<?php
namespace App\Helpers;

use Exception;

class PayUtils
{
    /**
     * Verify signature
     *
     * @param array $params parameter array (including sign and signType)
     * @param string $key key
     * @return bool
     */
    public static function verifySign(array $params, string $key): bool
    {
        if (!isset($params['sign']) || empty(trim($params['sign']))) {
            throw new Exception("Missing 'sign' field in parameters.");
        }

        if (!isset($params['signType']) || empty(trim($params['signType']))) {
            throw new Exception("Missing 'signType' field in parameters.");
        }

        $sign = strtoupper($params['sign']);
        $signType = strtoupper($params['signType']);

        // Copy the parameters and remove the sign field
        $paramsCopy = $params;
        unset($paramsCopy['sign']);

        $expectedSign = self::getSign(self::getSignStr($paramsCopy, $key), $signType);
        return $sign === $expectedSign;
    }

    /**
     * Sign the parameters
     *
     * @param array $params original parameters
     * @param string $key key
     * @return array Signed parameter array
     */
    public static function sign(array $params, string $key): array
    {
        if (!isset($params['signType']) || empty(trim($params['signType']))) {
            throw new Exception("Missing 'signType' field in parameters.");
        }

        $signType = strtoupper($params['signType']);
        $signStr = self::getSignStr($params, $key);
        $signValue = self::getSign($signStr, $signType);

        $params['sign'] = $signValue;
        return $params;
    }

    /**
     * Calculate signature value based on signature type
     *
     * @param string $signStr String to be signed
     * @param string $signType signature type (MD5/SHA1/SHA256)
     * @return string
     */
    public static function getSign(string $signStr, string $signType): string
    {
        switch ($signType) {
            case 'MD5':
                return strtoupper(md5($signStr));
            case 'SHA1':
                return strtoupper(sha1($signStr));
            case 'SHA256':
                return strtoupper(hash('sha256', $signStr));
            default:
                throw new Exception("Unsupported signature type: " . $signType);
        }
    }

    /**
     * Construct the string to be signed (exclude the sign field and sort by key)
     *
     * @param array $params parameter array
     * @param string $key key
     * @return string
     */
    public static function getSignStr(array $params, string $key): string
    {
        $sortedParams = self::sortValueRecursively($params);
        $items = [];

        foreach ($sortedParams as $k => $v) {
            if (strtolower($k) === 'sign') {
                continue;
            }

            if ($v === null || (is_string($v) && trim($v) === '')) {
                continue;
            }

            if (is_array($v)) {
                $v = json_encode($v, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            }

            $items[] = "{$k}={$v}";
        }

        $items[] = "key={$key}";
        return implode('&', $items);
    }

    /**
     * Recursively sort array or associative array
     *
     * @param mixed $value
     * @return mixed
     */
    public static function sortValueRecursively($value)
    {
        if (is_array($value)) {
            ksort($value); // Sort by key
            foreach ($value as &$item) {
                $item = self::sortValueRecursively($item);
            }
        }

        return $value;
    }
}